/*------------------------------------------------------------------------------*
 * Version - 8.0                                                                *
 * BY - Dilum Bandara                                                           *
 * Type - 									*
 * 	HHC									*
 * 	Breadth First tree formation						*
 * 	hierarchical naming					              	*
 * 	Consider node collisions						*
 * 	Nodes forward cluster formation message based on receiver power		*
 * 	A node out of cluster will not respond back to CH if it see a neighbor	*
 *	sending such an ACK back to CH						*
 *      Use kep-predisreibution                                                 *
 *------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include "simulator.h"
#include "energy.h"
#include "kps.h"

node nodes[NODESX][NODESY];			//Hold the node information
event *root;					//Root of the CH event list
char msg[100];
uint next_cid = 1;
packet data_packet;
uint collision_nodes[NO_COLLISION_NODES];	//List of nodes in the collision region
uint no_collision_nodes = 0;			//Number of nodes in the collision region
uint last_collision_set[2] = {0, 0};		//Last 2 nodes that caused the collision
uint last_type3_event_time = 0;
uchar optimized = 0;
float R;
key_index node_key_list[NONODES];

int main() {
    
    init(USE_NODE_FILE);			//initialize nodes & link list. Set to 1 to use the pregenerated node file
    while(1)                                    //Process the event list until no events are found
    {
        process_event_list();
    }
    return 0;
}

/*------------------------------------------------------------------------------*
 * Initialise the nodes & event list                                            *
 * Nodes can be either randomly genrated or assigned based on a pregenerated 	*
 * file. 									*
 * use_file - if 1 use the pregenerated node id file, else place nodes randomly	*
 *------------------------------------------------------------------------------*/
void init(uchar use_file) {
    int count, x, y, i, j, start_NID, NID;
    FILE *nodes_fd;
    char str[10];
    Hie_CID root_H_CID;
        
    count = 0;
    srand(time(NULL));                      //Set the seed for rand()
    R = transmission_range(0);              //Determine theoritical communication range based on low power
    
    //Set all the nodes to 0
    for (i = 0; i < NODESX ; i++) {
        for(j = 0 ; j < NODESY ; j++) {
            nodes[i][j].CID = 0;                    // Initially set all nodes to 0
            nodes[i][j].NID = 0;                    // Set NIDs from left to right. Set nos from 1
            nodes[i][j].no_broadcasts = 0;
            nodes[i][j].no_ACKs = 0;
            nodes[i][j].CH_NID = 0;
            nodes[i][j].tree_depth = 0;
            nodes[i][j].node_depth = 0;
            nodes[i][j].parent_CH_NID = 0;
            nodes[i][j].no_child_nodes = 0;
            nodes[i][j].no_routing_entries = 0;
            nodes[i][j].no_msg_forward = 0;
            nodes[i][j].sent_inform_neighbor = 0;
            nodes[i][j].node_dead = 0;
            nodes[i][j].energy = 0;
            nodes[i][j].marked_bcast_by_CID = 0;
            nodes[i][j].last_bcast_for_CID = 0;
            nodes[i][j].heard_ACK_for_CID = 0;
            nodes[i][j].send_ACK_for_CID = 0;
            nodes[i][j].send_tree_opt_msg = 255;
            nodes[i][j].no_CCHs = 0;
            nodes[i][j].no_key_to_CH = 0;
            nodes[i][j].no_neighbors = 0;
            nodes[i][j].no_key_neighbors = 0;
        }
    }

    if(use_file == 0) 	//if nodes are to be generated randomly
    {
        while (1)	//Loop forever
        {
            x = rand() % NODESX;
            y = rand() % NODESY;
            if(nodes[x][y].NID == 0) {
                nodes[x][y].NID = y * NODESX + x + 1;   // Set NIDs from left to right. Set nos from 1
                nodes[x][y].energy = E_NODE;		//Set the node's initial energy
                count++;
            }
            if(count == NONODES)		//Break if require no of nodes are generated
                break;
        }
    }
    else			//if previously generated node file to be used
    {
        nodes_fd = fopen(NODELIST, "r");			//Open file in read-only mode
        if(nodes_fd == NULL)					//If file is not open print error message
        {
            perror("ERROR:");
            exit(1);
        }

        for(i = 0; i < 	NONODES; i++)		//Read each node id from file
        {
            fgets(str, 10, nodes_fd);
            NID = atoi(str);
            x = (NID - 1) % NODESX;
            y = (NID - 1) / NODESX;
            nodes[x][y].NID = y * NODESX + x + 1;   	// Set NIDs from left to right on the grid. Set nos from 1
            nodes[x][y].energy = E_NODE;			//Set the nodes initial energy
        }

        fclose(nodes_fd);					//Close the file
    }

    if(nodes[STARTX][STARTY].NID == 0)	//See whether the starting node exist
    {
        printf("Can't continue. Intial node doesn't exist\n");
        exit(1);
    }

    //Add the first event to the event list
    start_NID = STARTY * NODESX + STARTX + 1;			//Determine the NID of the root node
    root_H_CID.id[0] = 0;						//Hierarchical cluster ID of the root node
    root_H_CID.id[1] = 0;
    root_H_CID.id[2] = 0;
    root_H_CID.id[3] = 0;

    //set the 1st event of forming the cluster. Event type 1.
    //Set start time as 1, CID as 1, no CH_NID since this node is the CH, hierarchical ID is 0 for 1st cluster, depth 0, no parent CH_NID or CID
    add_event(1, 1, start_NID, 1, start_NID, root_H_CID, 0, 0, MAX_TTL, 0, 0, root_H_CID);


    if(USE_KPS == 1)			//If key predistribution to be used	
        assign_key_id();		//then assign keys nodes
    print_no_common_keys();		//determine no of common keys each node has with its neighbor

/*
    if((USE_KPS == 1) && (NODE_COMP == 1))	//If key predistribution to be used & nodes are compromized
    {
        count = 0;
        while(1)				//Loop untill all the nodes are added
        { 
            x = rand() % NODESX;		//Randomly pick a x coordinate
            y = rand() % NODESY;		//randomly pick a y coordinate

            if(nodes[x][y].NID == 0)            //If the node has already not being placed	
                continue;
            else
            {
                for(z = 0; z < SC ; z++)
                    key_node.key_IDs[z] = nodes[x][y].key_IDs[z];

                retvalue = add_compromized_blocks(key_node, kps_p, kps_k, kps_z);
                if(retvalue == 1)
                        count++;
                else
                {
                    printf("Unable to add more compromized nodes\n. Terminating.....");
                    exit(1);
                }
            }
            if(count == COMP_NODES)
            break;
        }
    }
*/
}

/*------------------------------------------------------------------------------*
 * Add events to event list			                                *
 * Implementation of this function will slightly vary depending on the tree     *
 * formation approach (breadth-first, depath first, etc.)                       *
 * This implementation is for the Breadth-first                                 *
 * event_type	- 	Intercluster or Intracluster event			*
 * start time 	-	starting time of the event. events are sorted		*
 * nid		- 	node ID of the event related node			*
 * cid 		- 	CID of the event related node				*
 * ch_nid	- 	CH NID. This will be 0 if the event is related to a new	*
 * 			cluser formation. Will be > 0 if event is within cluster*
 * h_cid	- 	Hierarchical cluster ID of the cluster			*
 * tree_depth	- 	depth of the node broadcasting the message based on 	*
 * 			cluster tree						*
 * node_depth	- 	depth of the node broadcasting the message		*
 * ttl		-	current TTL value of the cluster formation broadcast	*
 * parent_cid	-	CID of the parent CH					*
 * parent_ch_nid-	NID of the parent CH					*
 * parent_h_cid	- 	Hierarchical CID of the parent				*
 *------------------------------------------------------------------------------*/
void add_event(uchar event_type, int start_time, uint nid, uint cid, uint ch_nid, Hie_CID h_cid, uchar tree_depth, uchar node_depth, uchar ttl, uint parent_cid, uint parent_ch_nid, Hie_CID parent_h_cid) {
    event *new, *current, *previous;

    if(root == NULL)				//If root of the event list is not defined
    {
        root = (event *)malloc(sizeof(event));
        if(root == NULL)			//If unable to allocate memory
        {
            perror("Error while allocating memory - in add_event function\n");
            exit(1);
        }
        root->next = NULL;
    }

    new = (event *)malloc(sizeof(event));		//New event
    if(new == NULL) {
        perror("Error while allocating memory - in add_event function\n");
        exit(1);
    }

    //Set the parameters for the new event
    new->event_type = event_type;
    new->time = start_time;
    new->NID = nid;
    new->CID = cid;
    new->CH_NID = ch_nid;
    new->H_CID = h_cid;
    new->tree_depth = tree_depth;
    new->node_depth = node_depth;
    new->TTL = ttl;
    new->parent_CID = parent_cid;
    new->parent_CH_NID = parent_ch_nid;
    if(parent_ch_nid != 0)
        new->parent_H_CID = parent_h_cid;

    if(root->next == NULL)					//if event list is blank
    {
        root->next = new;
        new->next = NULL;
    }
    else							//if event list contains at least one item
    {
        previous = root;
        current = root->next;

        while(1) {
            if(current->time > start_time) {
                new->next = current;
                previous->next = new;
                break;
            }
            else if(current->next == NULL) {
                current->next = new;
                new->next = NULL;
                break;
            }
            else {
                previous = current;
                current = current->next;
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Follwoing function remove the given event from the event list		*
 * start_time	-	remove the event related to the given start time	*
 * NID		- 	NID of the node related to the event			*
 *------------------------------------------------------------------------------*/
void remove_event(uchar event_type, uint start_time, uint nid) {
    event  *current, *previous;

    if(root->next == NULL)						//If event list is already empty
    {
        printf("Event list is already empty\n");
        exit(1);
    }

    //Locate the event with given time & NID

    previous = root;
    current = root->next;

    while (1) {
        //If no more events is found break the loop
        if(current->next == NULL)
            break;

        //if exact event is found break the loop
        if((current->event_type == event_type) && (current->time == start_time) && (current->NID == nid))
            break;

        previous = current;
        current = current->next;
    }

    if(current->next != NULL)					//If event is the last event
    {
        previous->next = current->next;
        free(current);
    }
    else								//if event is the 1st event or in the middle of the list
    {
        previous->next = NULL;
        free(current);
    }
}

/*------------------------------------------------------------------------------*
 * Process the event list starting from the first event, one event at a time	*
 * If no more events are available, print data about nodes & exit.		*
 *------------------------------------------------------------------------------*/
void process_event_list() {
    event *next_event;
    uint x0, y0, parentx0, parenty0, parent_nid, next_route_no, current_time, new_timeout;
    //uchar result;
    //uint no_msg_delivered = 0;	//No of sucessfully delivered messages
    //uint no_msg_dropped = 0;        //Nof of messages dropped
    //uint no_route = 0;		//No fo routes not found
    Hie_CID	tmp_h_cid;

    tmp_h_cid.id[0] = 0;
    tmp_h_cid.id[1] = 0;
    tmp_h_cid.id[2] = 0;
    tmp_h_cid.id[3] = 0;

    if(root->next == NULL)		//If no more events to handle print the data & exit
    {        
        //Test data delivery
        /*while (1)			//Loop forever
         * {
         * result = send_data();	//Send data from random source to a random destination
         * if(result == 1)		//If the message is dropped due to low energy
         * {
         * no_msg_dropped++;
         * break;
         * }
         * else if (result == 2)	//If message drop due to the wrong route
         * {
         * no_route++;
         * break;
         * }
         * else
         * no_msg_delivered++;
         * }*/

        //printf("%d\n", no_msg_delivered);
        //printf("%d\t%d\t%d\n", no_msg_delivered, no_msg_dropped, no_route);
        //test_routing();

        if(optimized == 0) 
        {
          //  print_no_common_keys();		//dump no of common keys each node has with its neighbor
            printf("%d\t%d\n", nodes_without_CH(0), nodes_without_CH(1));
            calculate_circularity(1);	//Dump node circularity info to file
            print_nodes(SHOW_NODE_DATA, 1);			//Dump node information to file
            //print_cluster_energy();	//Dump node energy info to file

            //opti_none_cluster_nodes();
            optimized = 1;
            return;
        }
        //printf("%d\t%d\n", nodes_without_CH(0), nodes_without_CH(1));
        //calculate_circularity(2);	//Dump node circularity info to file
        //print_nodes(SHOW_NODE_DATA, 2); //Dump node information to file        
        exit(0);
    }
    else 
    {       
        next_event = root->next;			//Get the event at the top of the event list
        current_time = next_event->time;		//Set the current time to the time of the event
        x0 = (next_event->NID - 1) % NODESX;		//x coordinates of node related to event
        y0 = (next_event->NID - 1) / NODESX;		//y coordinates of node related to event

        if(next_event->event_type == 1)			//event related to a node forwarding a cluster formation broadcast. Intracluster event
        {
            if(next_event->next != NULL)		//If there are at least 2 events
            {
                //if two nodes are broadcasting at the same time & if collisions to be considered
                if((next_event->time == next_event->next->time) && (USE_COLLISIONS == 1)) {
                    mark_collision_region(next_event->NID, next_event->next->NID); //Mark the collision region
                }
                else if(last_collision_set[1] != next_event->NID) //If the node was not in a collision region
                {
                    no_collision_nodes = 0;
                }
            }

            if(MAX_TTL == next_event->TTL)	//If node is the cluster head
            {
                if(nodes[x0][y0].CID == 0)     	//if not assigned to another cluser
                {
                    if(next_event->tree_depth != 254)	//if not a node trying to form a cluster by it self
                        add_nodes_to_cluster(current_time, next_event->NID, next_event->CID, next_event->NID, next_event->tree_depth, next_event->node_depth, next_event->TTL);
                    else
                        add_nodes_to_cluster(current_time, next_event->NID, next_event->CID, next_event->NID, next_event->tree_depth, next_event->node_depth,  1);

                    nodes[x0][y0].last_bcast_for_CID = next_event->CID;
                    nodes[x0][y0].no_broadcasts++;
                    if((optimized == 0) && (nodes[x0][y0].no_child_nodes > 0))	//If abel to attract any child node
                    {
                        nodes[x0][y0].CID = next_event->CID;            //set the cid of the starting node
                        nodes[x0][y0].H_CID = next_event->H_CID;
                        nodes[x0][y0].CH_NID = next_event->NID;         //set me as my own ch
                        nodes[x0][y0].tree_depth = next_event->tree_depth;
                        nodes[x0][y0].node_depth = next_event->node_depth;
                        nodes[x0][y0].parent_CH_NID = next_event->parent_CH_NID;

                        if(next_event->parent_CH_NID != 0)              //if not the root node
                        {
                            nodes[x0][y0].no_routing_entries++;
                            nodes[x0][y0].routing_table[0].valid = 3;       //valid & parent routing entry
                            nodes[x0][y0].routing_table[0].nei_H_CID = next_event->parent_H_CID;
                            nodes[x0][y0].routing_table[0].nei_NID = next_event->parent_CH_NID;

                            //add to parent routing table
                            parent_nid = nodes[x0][y0].parent_CH_NID;
                            parentx0 = (parent_nid - 1) % NODESX;
                            parenty0 = (parent_nid - 1) / NODESX;
                            next_route_no = nodes[parentx0][parenty0].no_routing_entries;
                            nodes[parentx0][parenty0].routing_table[next_route_no].valid = 1;
                            nodes[parentx0][parenty0].routing_table[next_route_no].nei_H_CID = next_event->H_CID;
                            nodes[parentx0][parenty0].routing_table[next_route_no].nei_NID = next_event->NID;
                            nodes[parentx0][parenty0].no_routing_entries++;
                        }

                        //Add a timeout event for this CH
                        new_timeout = current_time + TIMEOUT;
                        add_event(2, new_timeout, nodes[x0][y0].NID, nodes[x0][y0].CID, nodes[x0][y0].CH_NID, nodes[x0][y0].H_CID, nodes[x0][y0].tree_depth, nodes[x0][y0].node_depth, 0, 0, 0, tmp_h_cid);
                    }
                    else if(optimized == 1)				//Allow single node clusters in potimization phase
                    {
                        nodes[x0][y0].CID = next_event->CID;            //set the cid of the starting node
                        nodes[x0][y0].H_CID = next_event->H_CID;
                        nodes[x0][y0].CH_NID = next_event->NID;         //set me as my own ch
                        nodes[x0][y0].tree_depth = next_event->tree_depth;
                        nodes[x0][y0].node_depth = next_event->node_depth;
                        nodes[x0][y0].parent_CH_NID = next_event->parent_CH_NID;
                    }
                }
                remove_event(1, current_time, next_event->NID);		//remove event
            }
            else if((MAX_TTL - next_event->TTL) < MAX_HOPS)             //If bcast message is within MAX_HOPS add receiving nodes to cluster
            {
                if(nodes[x0][y0].CID == next_event->CID) 		      	//if not assigned to another cluser allow to add more nodes
                {
                    if(nodes[x0][y0].last_bcast_for_CID != next_event->CID) 
                    {
                        add_nodes_to_cluster(current_time, next_event->NID, next_event->CID, next_event->CH_NID, next_event->tree_depth, next_event->node_depth, next_event->TTL);
                        nodes[x0][y0].last_bcast_for_CID = next_event->CID;
                        nodes[x0][y0].no_broadcasts++;
                    }
                }
                remove_event(1, current_time, next_event->NID);			//remove event
            }
            else if((MAX_TTL - next_event->TTL) < (MAX_TTL))			//if out side cluster
            {
                if(nodes[x0][y0].marked_bcast_by_CID == next_event->CID)       	//if assigned to be bcasted by for the same cluster
                {
                    if(nodes[x0][y0].last_bcast_for_CID != next_event->CID)
                    {
                        forward_bcast_cluster(current_time, next_event->NID, next_event->CID, next_event->CH_NID, next_event->TTL, next_event->node_depth);
                        nodes[x0][y0].last_bcast_for_CID = next_event->CID;
                        nodes[x0][y0].no_broadcasts++;
                    }
                }
                remove_event(1, current_time, next_event->NID);			//remove event
            }
            else							//if TTL expired send ACK to CH
            {
                //if assigned to be bcasted by for the same cluster & not already heard an ACK from a neighbor
                if(nodes[x0][y0].marked_bcast_by_CID == next_event->CID)
                    send_ACK_as_CCH(next_event->NID, next_event->CID, next_event->CH_NID);

                remove_event(1, current_time, next_event->NID);	//remove event
            }
        }
        else if (next_event->event_type == 2)		//event related to a CH timeout
        {
            if(nodes[x0][y0].no_CCHs > 255)
            {
                printf("Error: No of CCHs > 255. Overflow\n");
            }
            select_child_CHs(current_time, nodes[x0][y0].no_CCHs, next_event->CID, next_event->CH_NID, next_event->H_CID, next_event->tree_depth, next_event->node_depth);
            remove_event(2, current_time, next_event->NID);		//remove event
        }
    }
}

/*------------------------------------------------------------------------------*
 * Add nodes to cluster. 							*
 * Mark all the nodes in the communication range of the broadcasting node & 	*
 * without a cluster. In 1-hop cluster nid == ch_nid but this will be different	*
 * in multi-hop clusters.							*
 * start_time	- 	start time of the current event				*
 * nid		- 	NID of the node sending the bcast			*
 * cid		-	CID of the new clusters					*
 * ch_nid	-	NID of the CH						*
 * depth	- 	depth of the node sending the bcast			*
 * ttl		- 	TTL of the bcast					*
 *------------------------------------------------------------------------------*/
void add_nodes_to_cluster(uint start_time, uint nid, uint cid, uint ch_nid, uchar tree_depth, uchar node_depth, uchar ttl) 
{
    int x0, y0, minx, miny, maxx, maxy, ch_x, ch_y, l, k, retvalue, z;
    float distance;						//distance between reciveing & transmitting node
    char rec_rssi;
    uchar new_tree_depth, new_node_depth, new_ttl;
    uint new_start_time;
    Hie_CID tmp_h_cid;
    region my_region;
    key_index key_CH, key_child; 
    
    x0 = (nid - 1) % NODESX;
    y0 = (nid - 1) / NODESX;

    my_region = get_node_region(x0, y0, R);	//get my region
    minx = my_region.minx;
    miny = my_region.miny;
    maxx = my_region.maxx;
    maxy = my_region.maxy;

    ch_x = (ch_nid - 1) % NODESX;		//x coordinates of ch
    ch_y = (ch_nid - 1) / NODESX;		//y coordinates of ch

    if(USE_KPS == 1)				//If KPS is used
    {
        for(z = 0; z < M ; z++)
            key_CH.key_IDs[z] = node_key_list[nodes[ch_x][ch_y].key_pointer].key_IDs[z];
    }
    
    tmp_h_cid.id[0] = 0;
    tmp_h_cid.id[1] = 0;
    tmp_h_cid.id[2] = 0;
    tmp_h_cid.id[3] = 0;

    new_tree_depth = tree_depth + 1;				//As the message travles depth increases & TTL reduces
    new_node_depth = node_depth + 1;
    new_ttl = ttl - 1;

    //set cid for only neighboring nodes
    for(l = miny; l <= maxy; l++) 
    {
        for (k = minx; k <= maxx; k++) 
        {
            distance = sqrt((k - x0)*(k-x0)*(GRIDX * GRIDX) + (l - y0)*(l-y0)*(GRIDY * GRIDY));
            rec_rssi = RSSI(distance, 0);     //determine the RSSI for the signal for the given distance
            // if within communication range & if the node exist & not the same node (distance != 0)
           
            if ((rec_rssi >= 0) && (nodes[k][l].NID != 0) && (distance != 0)) 
            {
                //nodes[k][l].energy -= energy_to_receive(cluster_bcast_size);	//energy consumed to receive a message

                //if not a member of a cluster or not in the collision region assigned to the current cluster
                if((nodes[k][l].CID == 0) && (is_in_collision(nodes[k][l].NID) == 0)) 
                {
                    if(USE_KPS == 1) 	//if key predistribution scheme to be used
                    {
                        for(z = 0; z < M ; z++)
                            key_child.key_IDs[z] = node_key_list[nodes[k][l].key_pointer].key_IDs[z];
                          
                        retvalue =  common_key(key_CH, key_child, &(nodes[k][l].common_key_ID));
                        if(retvalue == 1)	//If a common key exist
                        {
                            nodes[k][l].CID = cid;
                            nodes[k][l].CH_NID = ch_nid;
                            nodes[k][l].tree_depth = new_tree_depth;
                            nodes[k][l].node_depth = new_node_depth;
                            nodes[ch_x][ch_y].no_child_nodes++;
                            nodes[k][l].no_ACKs++;

                            new_start_time = start_time + random_wait_time();
                            if(new_start_time <= start_time)
                                new_start_time = start_time + 1;
                            //Add to event list based on RSSI value. Add as type 1 event
                            if((new_ttl > 0) && (nodes[k][l].marked_bcast_by_CID != cid))	//if TTL has not expired
                            {
                                add_event(1, new_start_time, nodes[k][l].NID, cid, ch_nid, tmp_h_cid, new_tree_depth, new_node_depth, new_ttl, 0, 0, tmp_h_cid);
                                nodes[k][l].marked_bcast_by_CID = cid;
                            }
                        }
                        else
                            nodes[k][l].no_key_to_CH = 1;	//unable to join cluster due to no common key
                    }
                    else
                    {
                        nodes[k][l].CID = cid;
                        nodes[k][l].CH_NID = ch_nid;
                        nodes[k][l].tree_depth = new_tree_depth;
                        nodes[k][l].node_depth = new_node_depth;
                        nodes[ch_x][ch_y].no_child_nodes++;
                        nodes[k][l].no_ACKs++;

                        new_start_time = start_time + random_wait_time();
                        if(new_start_time <= start_time)
                            new_start_time = start_time + 1;
                        //Add to event list based on RSSI value. Add as type 1 event
                        if((new_ttl > 0) && (nodes[k][l].marked_bcast_by_CID != cid))	//if TTL has not expired
                        {
                            add_event(1, new_start_time, nodes[k][l].NID, cid, ch_nid, tmp_h_cid, new_tree_depth, new_node_depth, new_ttl, 0, 0, tmp_h_cid);
                            nodes[k][l].marked_bcast_by_CID = cid;
                        }
                    }
                }
            }
        }
    }
}

/*------------------------------------------------------------------------------*
 * Forward the cluster fomation bcast                                           *
 * Consider all the nodes in the communication range of the bcasting node &     *
 * without a cluster.								*
 * start_time   -       start time of the current event                         *
 * nid          -       NID of the node sending the bcast                       *
 * cid          -       CID of the new clusters                                 *
 * ch_nid       -       NID of the CH                                           *
 * ttl          -       TTL of the bcast                                        *
 *------------------------------------------------------------------------------*/
void forward_bcast_cluster(uint start_time, uint nid, uint cid, uint ch_nid, uchar ttl, uchar node_depth) {
    int x0, y0, minx, miny, maxx, maxy, ch_x, ch_y, l, k, z, retvalue;
    float distance;						//distance between reciveing & transmitting node
    uchar new_ttl;
    char rec_rssi;
    uint new_start_time;
    Hie_CID tmp_h_cid;
    region my_region;
    float connectivity;
    key_index key_CH, key_cch;

    x0 = (nid - 1) % NODESX;
    y0 = (nid - 1) / NODESX;

    my_region = get_node_region(x0, y0, R);			//Get my region
    minx = my_region.minx;
    miny = my_region.miny;
    maxx = my_region.maxx;
    maxy = my_region.maxy;

    ch_x = (ch_nid - 1) % NODESX;		//x coordinates of ch
    ch_y = (ch_nid - 1) / NODESX;		//y coordinates of ch

    tmp_h_cid.id[0] = 0;
    tmp_h_cid.id[1] = 0;
    tmp_h_cid.id[2] = 0;
    tmp_h_cid.id[3] = 0;
    
    new_ttl = ttl - 1;				//As the message travles TTL reduces
    
    if(USE_KPS == 1)				//If KPS is used
    {
        for(z = 0; z < M ; z++)
            key_CH.key_IDs[z] = node_key_list[nodes[ch_x][ch_y].key_pointer].key_IDs[z];
    }   
    //set cid for only neighboring nodes
    for(l = miny; l <= maxy; l++) 
    {
        for (k = minx; k <= maxx; k++) 
        {
            distance = sqrt((k - x0)*(k-x0)*(GRIDX * GRIDX) + (l - y0)*(l-y0)*(GRIDY * GRIDY));
            rec_rssi = RSSI(distance, 0);  //determine the RSSI for the signal for the given distance
            
            // if within communication range & if the node exist & not the same node (distance != 0)
            if ((rec_rssi >= 0) && (nodes[k][l].NID != 0) && (distance != 0))
            {
                //nodes[k][l].energy -= energy_to_receive(cluster_bcast_size);	//energy consumed to receive a message

                //if not a member of a cluster & not used to send the same bcast use it to bcast the message
                if((nodes[k][l].CID == 0) && (nodes[k][l].marked_bcast_by_CID != cid) && (is_in_collision(nodes[k][l].NID) == 0)) 
                {  
                    if(new_ttl > 0)
                        new_start_time = start_time + random_wait_time();
                    else
                    {
                        //See whether it has a key to the parent CH
                        for(z = 0; z < M ; z++)
                            key_cch.key_IDs[z] = node_key_list[nodes[x0][y0].key_pointer].key_IDs[z];

                        retvalue =  common_key(key_CH, key_cch, &(nodes[x0][y0].common_key_ID));
                        if(retvalue == 1)	//If a common key exist
                        {
                            if(nodes[k][l].no_neighbors > 0)
                                connectivity = (float) nodes[k][l].no_key_neighbors/nodes[k][l].no_neighbors;
                            else    
                                continue;
                            if(connectivity > 0)
                                new_start_time = start_time + (int)(1/connectivity * 10 * MAX_RND_TIME) + random_wait_time();                        
                            else
                                continue;                        
                        }
                        else
                            continue;
                    }
                    if(new_start_time <= start_time)
                        new_start_time = start_time + 1;
                    //Add to event list based on RSSI value. Add as type 1 event
                    add_event(1, new_start_time, nodes[k][l].NID, cid, ch_nid, tmp_h_cid, 0, (node_depth + 1), new_ttl, 0, 0, tmp_h_cid);
                    nodes[k][l].marked_bcast_by_CID = cid;
                }
            }
        }
    }
}

/*------------------------------------------------------------------------------*
 * Send an acknoledgement to the CH indicating that node is a candiadte to be a *
 * a new CH. Consider all the nodes in the communication range of the bcasting	*
 * node & without a cluster.							*
 * start_time   -       start time of the current event                         *
 * nid          -       NID of the node sending the bcast                       *
 * cid          -       CID of the new clusters                                 *
 * ch_nid       -       NID of the CH                                           *
 * ttl          -       TTL of the bcast                                        *
 *------------------------------------------------------------------------------*/
void send_ACK_as_CCH(uint nid, uint cid, uint ch_nid) 
{
    int x0, y0, minx, miny, maxx, maxy, ch_x, ch_y, l, k, i, retvalue, z;
    float distance;				//distance between reciveing & transmitting node
    uint tmp_no_CCHs;
    region my_region;
    char rec_rssi;
    key_index key_CH, key_cch;
  
    x0 = (nid - 1) % NODESX;
    y0 = (nid - 1) / NODESX;
    ch_x = (ch_nid - 1) % NODESX;		//x coordinates of ch
    ch_y = (ch_nid - 1) / NODESX;		//y coordinates of ch

    if(USE_KPS == 1)				//If KPS is used
    {
        for(z = 0; z < M ; z++)
            key_CH.key_IDs[z] = node_key_list[nodes[ch_x][ch_y].key_pointer].key_IDs[z];
    }

    //Makesure that node has not already send a ACK for the same cluster or heard an ACK from a neighbor
    if((nodes[x0][y0].send_ACK_for_CID != cid) && (nodes[x0][y0].heard_ACK_for_CID != cid)) 
    {
        for(i = 0; i < nodes[ch_x][ch_y].no_CCHs; i++) 
        {
            if(nid == nodes[ch_x][ch_y].CCHs[i])
                break;
        }
        if(i ==  nodes[ch_x][ch_y].no_CCHs) 
        {
            if(USE_KPS == 1) 	//if key predistribution scheme to be used
            {
                for(z = 0; z < M ; z++)
                    key_cch.key_IDs[z] = node_key_list[nodes[x0][y0].key_pointer].key_IDs[z];

                retvalue =  common_key(key_CH, key_cch, &(nodes[x0][y0].common_key_ID));
                if(retvalue == 1)	//If a common key exist
                {
                    tmp_no_CCHs = nodes[ch_x][ch_y].no_CCHs;
                    nodes[ch_x][ch_y].CCHs[tmp_no_CCHs] = nid;
                    nodes[ch_x][ch_y].no_CCHs = tmp_no_CCHs + 1;
                    nodes[x0][y0].send_ACK_for_CID = cid;
                    nodes[x0][y0].heard_ACK_for_CID = cid;
                    nodes[x0][y0].no_ACKs = nodes[x0][y0].no_ACKs + 3;  //ACK forwarded by 3-hops
                }
                else
                    return;
            }
            else
            {
                tmp_no_CCHs = nodes[ch_x][ch_y].no_CCHs;
                nodes[ch_x][ch_y].CCHs[tmp_no_CCHs] = nid;
                nodes[ch_x][ch_y].no_CCHs = tmp_no_CCHs + 1;
                nodes[x0][y0].send_ACK_for_CID = cid;
                nodes[x0][y0].heard_ACK_for_CID = cid;
                nodes[x0][y0].no_ACKs = nodes[x0][y0].no_ACKs + 3;  //ACK forwarded by 3-hops    
            }
        }
        else
            return;
    }
    else							//if so discard
        return;

    my_region = get_node_region(x0, y0, R);			//get my region
    minx = my_region.minx;
    miny = my_region.miny;
    maxx = my_region.maxx;
    maxy = my_region.maxy;

    //set cid for only neighboring nodes
    for(l = miny; l <= maxy; l++) 
    {
        for (k = minx; k <= maxx; k++) 
        {
            distance = sqrt((k - x0)*(k-x0)*(GRIDX * GRIDX) + (l - y0)*(l-y0)*(GRIDY * GRIDY));
            rec_rssi = RSSI(distance, 0);
            
            // if within communication range & if the node exist & not the same node (distance != 0)
            if ((rec_rssi >= 0) && (nodes[k][l].NID != 0) && (distance != 0)) {
                //nodes[k][l].energy -= energy_to_receive(cluster_bcast_size);	//energy consumed to receive a message

                //if not a member of a cluster & marked to forward the bcast
                if((nodes[k][l].CID == 0) && (nodes[k][l].marked_bcast_by_CID == cid) && (is_in_collision(nodes[k][l].NID) == 0)) 
                {
                    nodes[k][l].heard_ACK_for_CID = cid;
                }
            }
        }
    }
}


/*------------ -----------------------------------------------------------------*
 * Following function selects candidate CHs for list of available CHs		*
 * no_cchs	- 	no of CCHs						*
 * depth	- 	depth of the parent CH					*
 * parent_cid	- 	CID of the parent CH					*
 * parent_h_cid	-	Hierarchical CID of the parent CH			*
 * parent_ch_nid-	NID of the parent CH					*
 *------------------------------------------------------------------------------*/
void select_child_CHs(uchar event_time, uchar no_cchs, uint parent_cid, uint parent_ch_nid, Hie_CID parent_h_cid, uchar tree_depth, uchar node_depth) {
    uint no_new_chs, next_ch, j, l, new_time, x0, y0, retvalue, child_x0, child_y0, z;
    uint selected_ch_list[10];
    uint no_selected_chs = 0;
    Hie_CID new_h_cid;
    key_index key_parent, key_child; 
    uchar loops = 0;				//no of key indexes in a node

    if(no_cchs == 0)			//if no nodes are there to be elected as CCHs just return back to caller
        return;

    x0 = (parent_ch_nid - 1) % NODESX;
    y0 = (parent_ch_nid - 1) / NODESX;

    if(USE_KPS == 1)					//if key predistribution scheme to be used
    {	
        for(z = 0; z < M ; z++)
            key_parent.key_IDs[z] = node_key_list[nodes[x0][y0].key_pointer].key_IDs[z];
    }
    
    if(tree_depth == 0)
        no_new_chs= NO_CCHS;
    else
        no_new_chs= NO_CCHS/2;

    if(no_new_chs > no_cchs)		//if no of possible CCHs are less than what is given in the header file
        no_new_chs = no_cchs;		//generate the maximum possible no

    for(j = 0; j < no_new_chs; j++)		//select given no of candidate neighbors as chs
    {
        while(1) {
            next_ch = rand() % no_cchs; 			//randomly select one of the nodes to be the ch
            for(l = 0 ; l < no_selected_chs ; l++)		//check whether it has been already selected
            {
                if(selected_ch_list[l] == nodes[x0][y0].CCHs[next_ch])	//if so don't selecte again
                    break;
            }

            if(l != no_selected_chs)				//if so diacard & select another
                continue;

            //if key predistribution scheme to be used don't select nodes with a common key with the parent CH
            if(USE_KPS == 1)				
            {
                child_x0 = (nodes[x0][y0].CCHs[next_ch] - 1) % NODESX;
                child_y0 = (nodes[x0][y0].CCHs[next_ch] - 1) / NODESX;
                for(z = 0; z < M ; z++)
                    key_child.key_IDs[z] = node_key_list[nodes[child_x0][child_y0].key_pointer].key_IDs[z];

                retvalue =  common_key(key_parent, key_child, &(nodes[child_x0][child_y0].common_key_ID));
                if(retvalue == 0)	//If no common key exist skip the node
                {
                    loops++;	//increment loop counter when common node is not found	
                    //200 is just an aribiatory value just to prevent the loop form becoming infinite
                    if(loops == 254) //Just keep looping until a possble node is found or until loop count exceeds.
                        break;
                    else
                        continue;
                }
            }
            
            selected_ch_list[no_selected_chs] = nodes[x0][y0].CCHs[next_ch];
            no_selected_chs++;

            next_cid++;						//generate the next cid

            new_h_cid = generate_CID(parent_h_cid, j, (tree_depth + 1));
            //set the timeing such that tree is formed using the breadth-first tree formation
            if (j == 0)
                new_time = last_event_time((tree_depth + 1), NO_CCHS, TIMEOUT, DELAY_CCH) + 1 ;
            else
                new_time = last_event_time((tree_depth + 1), NO_CCHS, TIMEOUT, DELAY_CCH) + j * DELAY_CCH ;

            if(new_time <= event_time)
                new_time = event_time + 1;
            //add to the event list
            add_event(1, new_time, nodes[x0][y0].CCHs[next_ch], next_cid, 0, new_h_cid, (tree_depth + 1), (node_depth + MAX_TTL), MAX_TTL, parent_cid, parent_ch_nid, parent_h_cid);
            nodes[x0][y0].no_broadcasts = nodes[x0][y0].no_broadcasts + 3;  //Bcast is send 3-hops
            break;
        }
    }
}

/*------------------------------------------------------------------------------*
 * follwing function calculates the maximum achievable circularity (mac) for    *
 * each cluster & print circularity to the circularity.txt file                 *
 * this functions works only up to 1-hop clusters. can be extened to 4-hops by  *
 * adding appropriate codes                                                     *
 *------------------------------------------------------------------------------*/
void calculate_circularity(uchar file) {
    int i, j, x0, y0, minx, miny, maxx, maxy, l, k, retvalue;
    float length, in_cluster, outof_cluster, circularity;
    int nb_list[3][500];
    int level0, level1, level2, n, m, p, q;
    region my_region;

    FILE *circlefd;
    if(file == 2)
        circlefd = fopen(CIRCLEFILE2, "w");
    else
        circlefd = fopen(CIRCLEFILE1, "w");


    for (i = 0; i < NODESX ; i++) {
        for(j = 0 ; j < NODESY ; j++) {
            //make sure that the node is a ch
            if((nodes[i][j].NID == nodes[i][j].CH_NID) && (nodes[i][j].NID != 0)) {
                in_cluster = 0;				//no of nodes in side cluster
                outof_cluster = 0;			//no of nodes out side cluster
                level0 = level1 = level2 = 0;	//check the hope for hop-head & multi-hop clustering

                x0 = (nodes[i][j].NID - 1) % NODESX;
                y0 = (nodes[i][j].NID - 1) / NODESX;

                my_region = get_node_region(x0, y0, R);	//Get my region
                minx = my_region.minx;
                miny = my_region.miny;
                maxx = my_region.maxx;
                maxy = my_region.maxy;

                for(l = miny; l <= maxy; l++) {
                    for (k = minx; k <= maxx; k++) {
                        length = sqrt((k - x0)*(k-x0)*(GRIDX * GRIDX) + (l - y0)*(l-y0)*(GRIDY * GRIDY));
                        if((length <=  R) && (nodes[k][l].NID != 0))     // if within communication range
                        {
                            if(nodes[k][l].CID == nodes[i][j].CID)
                                in_cluster++;
                            else
                                outof_cluster++;
                            nb_list[0][level0] = nodes[k][l].NID;
                            level0++;
                        }
                    }
                }
                if( (MAX_HOPS > 1) && (level0 != 0)) {
                    for(n=0; n < level0 ; n++) {
                        x0 = (nb_list[0][n] - 1) % NODESX;
                        y0 = (nb_list[0][n] - 1) / NODESX;

                        my_region = get_node_region(x0, y0, R);		//get my region
                        minx = my_region.minx;
                        miny = my_region.miny;
                        maxx = my_region.maxx;
                        maxy = my_region.maxy;

                        for(l = miny; l <= maxy; l++) {
                            for (k = minx; k <= maxx; k++) {
                                length = sqrt((k-x0)*(k-x0)*(GRIDX * GRIDX) + (l-y0)*(l-y0)*(GRIDY * GRIDY));
                                if((length <=  R) && (nodes[k][l].NID != 0)) {
                                    for(m = 0; m < level0; m++) {
                                        if(nb_list[0][m] == nodes[k][l].NID)
                                            break;
                                    }
                                    for( p = 0; p < level1; p++) {
                                        if(nb_list[1][p] == nodes[k][l].NID)
                                            break;
                                    }
                                    if((m == level0) && (p == level1)) {
                                        if(nodes[k][l].CID == nodes[i][j].CID)
                                            in_cluster++;
                                        else
                                            outof_cluster++;
                                        nb_list[1][level1] = nodes[k][l].NID;
                                        level1++;
                                    }
                                }
                            }
                        }
                    }
                }
                if( (MAX_HOPS > 2) && (level1 != 0)) {
                    for(n=0; n < level1 ; n++) {
                        x0 = (nb_list[1][n] - 1) % NODESX;
                        y0 = (nb_list[1][n] - 1) / NODESX;

                        my_region = get_node_region(x0, y0, R);		//get my region
                        minx = my_region.minx;
                        miny = my_region.miny;
                        maxx = my_region.maxx;
                        maxy = my_region.maxy;

                        for(l = miny; l <= maxy; l++) {
                            for (k = minx; k <= maxx; k++) {
                                length = sqrt((k-x0)*(k-x0)*(GRIDX * GRIDX) + (l-y0)*(l-y0)*(GRIDY * GRIDY));
                                if((length <=  R) && (nodes[k][l].NID != 0)) {
                                    for(m = 0; m < level0; m++) {
                                        if(nb_list[0][m] == nodes[k][l].NID)
                                            break;
                                    }
                                    for( p = 0; p < level1; p++) {
                                        if(nb_list[1][p] == nodes[k][l].NID)
                                            break;
                                    }
                                    for(q= 0; q < level2; q++) {
                                        if(nb_list[2][q] == nodes[k][l].NID)
                                            break;
                                    }

                                    if((m == level0) && (p == level1) && (q == level2)) {
                                        if(nodes[k][l].CID == nodes[i][j].CID)
                                            in_cluster++;
                                        else
                                            outof_cluster++;
                                        nb_list[2][level2] = nodes[k][l].NID;
                                        level2++;
                                    }
                                }
                            }
                        }

                    }

                }

                circularity = (in_cluster/(in_cluster + outof_cluster)) * 100;
                //retvalue = sprintf(msg, "%d\t%f\n", nodes[i][j].CID, circularity);
                retvalue = sprintf(msg, "%f\n", circularity);
                fputs(msg, circlefd);
            }
        }
    }
    fclose(circlefd);	//close file
}



/*------------------------------------------------------------------------------*
 * Following function dump remaining energy of each node to a text file		*
 *------------------------------------------------------------------------------*/
void print_cluster_energy()
{
    int i, j, retvalue;
    FILE *energyfd;
    energyfd = fopen(ENERGYFILE, "w");

    //Following code write node energy to the text file
    for (i = 0; i < NODESY ; i++) {
        for(j = 0 ; j < NODESX ; j++) {
            if(nodes[j][i].NID != 0) {
                retvalue = sprintf(msg, "%f\n", nodes[j][i].energy);
                fputs(msg, energyfd);
            }
        }
    }
    fclose(energyfd);
}

/*------------------------------------------------------------------------------*
 * This function either printer the node status on the terminal or print node   *
 * data to a text file named nodes.txt                                          *
 * symbols:                                                                     *
 * '.' - Indicate grid points with nodes                                        *
 * 'o' - Indicate nodes with a CH                                               *
 * '?' - Indicare nodes clusters that don't have a represnetable symbol         *
 * Cluster symbol followed by a . indicate CHs                                  *
 *------------------------------------------------------------------------------*/
void print_nodes(uchar pnt_console, uchar file) 
{
    int i, j, retvalue;
    FILE *nodefd;
    if(file == 1)
        nodefd = fopen(NODEFILE1, "w");
    else
        nodefd = fopen(NODEFILE2, "w");


    //Following code write node data to the text file
    for (i = 0; i < NODESY ; i++) {
        for(j = 0 ; j < NODESX ; j++) {
            if(nodes[j][i].NID != 0) {
                retvalue = sprintf(msg, "%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\n", nodes[j][i].NID, nodes[j][i].CID, nodes[j][i].CH_NID, nodes[j][i].parent_CH_NID, nodes[j][i].tree_depth, nodes[j][i].no_child_nodes, nodes[j][i].no_broadcasts, nodes[j][i].no_ACKs, nodes[j][i].node_depth);
                fputs(msg, nodefd);
            }
        }
    }
    fclose(nodefd);

    if(pnt_console == 1) {
        for (i = 0; i < NODESY ; i++) {
            for(j = 0 ; j < NODESX ; j++) {
                if(nodes[j][i].NID == 0)
                    printf(". ");
                else {
                    printf("%c", CID_to_symbol_mapping(nodes[j][i].CID));

                    if(nodes[j][i].NID == nodes[j][i].CH_NID)
                        printf(".");
                    else
                        printf(" ");
                }
            }
            printf("\n");
        }
    }
}

/*------------------------------------------------------------------------------*
 * Following function add a node a collision list if its is in the communication*
 * range of two CHs at the same time.						*
 * nid1		- NID of the first CH node					*
 * nid2		- NID of the second CH node					*
 *------------------------------------------------------------------------------*/
void mark_collision_region(uint nid1, uint nid2)
{
    float ch_distance, distance1, distance2;
    int x1, y1, x2, y2, minx, maxx, miny, maxy, l, k, i;
    region my_region;

    if(last_collision_set[1] != nid1)	//Reset the list if last collision is not related
        no_collision_nodes = 0;

    last_collision_set[0] = nid1;		//Set the last 1 nodes related to the collision
    last_collision_set[1] = nid2;

    x1 = (nid1 - 1) % NODESX;		//x coordinates of node related to event
    y1 = (nid1 - 1) / NODESX;		//y coordinates of node related to event
    x2 = (nid2 - 1) % NODESX;		//x coordinates of node related to event
    y2 = (nid2 - 1) / NODESX;		//y coordinates of node related to event

    ch_distance = sqrt((x1 - x2)*(x1 - x2)*(GRIDX * GRIDX) + (y1 - y2)*(y1 - y2)*(GRIDY)*(GRIDY));
    if(ch_distance <= (2 * R))					//if within each others range
    {
        my_region = get_node_region(x1, y1, R);		//get my region
        minx = my_region.minx;
        miny = my_region.miny;
        maxx = my_region.maxx;
        maxy = my_region.maxy;

        for(l = miny; l <= maxy; l++) {
            for (k = minx; k <= maxx; k++) {
                distance1 = sqrt((k - x1)*(k - x1)*(GRIDX * GRIDX) + (l - y1)*(l - y1)*(GRIDY * GRIDY));
                if((distance1 <= R) && (nodes[k][l].NID != 0)) {
                    distance2 = sqrt((k - x2)*(k - x2)*(GRIDX * GRIDX) + (l - y2)*(l - y2)*(GRIDY * GRIDY));
                    if((distance2 <= R) && (nodes[k][l].NID != 0)) {
                        if(no_collision_nodes < NO_COLLISION_NODES)	//Check for overflow
                        {
                            for(i = 0; i < no_collision_nodes; i++)		//Don't put the same node again & again
                            {
                                if(collision_nodes[i] == nodes[k][l].NID)
                                    break;
                            }
                            if(i == no_collision_nodes)			//if no match found add
                            {
                                collision_nodes[no_collision_nodes] = nodes[k][l].NID;	//Add node to the collision region
                                no_collision_nodes++;			//Increment number of nodes in the collision region
                            }
                        }
                        else {
                            printf("No of collision nodes overflow...\n");
                            exit(0);
                        }
                    }
                }
            }
        }
    }
}

/*------------------------------------------------------------------------------*
 * Following function checks whether a given node is in the collision range	*
 * nid		- NID of the node						*
 * Return 	- 1 if in the collision region & 0 if not			*
 *------------------------------------------------------------------------------*/
char is_in_collision(uint nid) 
{
    int i;

    for(i = 0; i < no_collision_nodes; i++)		//See to see whether the node is in the collsion list
    {
        if(collision_nodes[i] == nid)		//if so break
            break;
    }
    if(i != no_collision_nodes)
        return 1;				//if in collision region
    else
        return 0;
}


/*----------------------------------------------------------------------*
 * Allow nodes that are not in a cluster to join neighboring cluster	*
 *----------------------------------------------------------------------*/
void opti_none_cluster_nodes() {
    int i, j, k, l, minx, maxx, miny, maxy;
    region my_region;
    float distance;
    uchar my_exit = 0;
    char rec_rssi;

    for (j = 0; j < NODESY ; j++) {
        for(i = 0 ; i < NODESX ; i++) {
            //if the node exist but not in a cluster
            if((nodes[i][j].NID != 0) && (nodes[i][j].CID == 0)) {
                my_exit = 0;

                my_region = get_node_region(i, j, R);		//get my region
                minx = my_region.minx;
                miny = my_region.miny;
                maxx = my_region.maxx;
                maxy = my_region.maxy;

                for(l = miny; l <= maxy; l++) {
                    for (k = minx; k <= maxx; k++) {
                        distance = sqrt((k - i)*(k - i)*(GRIDX * GRIDX) + (l - j)*(l - j)*(GRIDY * GRIDY));
                        rec_rssi = RSSI(distance, 0);
                        
                        // if within communication range & if the node exist & a CH
                        if((rec_rssi >= 0) && (nodes[k][l].NID != 0) && (nodes[k][l].NID == nodes[k][l].CH_NID)) {
                            nodes[i][j].CID = nodes[k][l].CID;
                            nodes[i][j].CH_NID = nodes[k][l].CH_NID;
                            nodes[i][j].tree_depth = nodes[k][l].tree_depth + 1;
                            nodes[i][j].node_depth = nodes[k][l].node_depth + 1;
                            nodes[i][j].no_ACKs++;
                            nodes[k][l].no_child_nodes++;
                            my_exit = 1;			//Exit both loops
                            break;
                        }
                    }
                    if(my_exit == 1)			//exit 1st outer loop
                        break;
                }
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Following function assign key indexes for node				*
 * It assigns only the key index not individual key IDs				*
 *------------------------------------------------------------------------------*/
void assign_key_id()
{
    FILE *key_fd;			//File descripter
    char msg[10];			//Hold the string read from the file
    int retvalue, tmp_count;
    uint i, j, k, x, y, p, q, rnd_id;
    uint tmp_key_id;                    //Holds the key id
    key_block key_list[10][10];		//List of key indexes
    int next_key_pointer = 0;

    key_fd = fopen(KEYFILE, "r");	//Open file as read only
    if(key_fd == NULL)		//If unable to open file
    {
        perror("Error- Key ID file: ");
        exit(1);
    }

    //Load list of key IDs
    for(i = 0; i < 10; i++)
    {
        for(j = 0; j < 10; j++)
        {
            for(k = 0; k < SC; k++)
            {
                fgets(msg, 10, key_fd);
                retvalue = sscanf(msg, "%d", &tmp_key_id);
                key_list[i][j].key_IDs[k] = tmp_key_id;    
            }           
        }
    }
    
    for(x = 0; x < NODESX ; x++)       
    {
        for(y = 0; y < NODESY; y++)
        {
            if(nodes[x][y].NID != 0)
            {                
                nodes[x][y].key_pointer = next_key_pointer;            
                tmp_count = 0;
                p = x / 100;
                q = y / 100;
                if(x == 1000)
                   p = 9;
                if(y == 1000)
                   q = 9;
             
                while(1)
                {
                   if(tmp_count == M)
                       break;
                    rnd_id = rand() % SC;                
                    for(k = 0; k < M; k++)
                    {
                        if(node_key_list[next_key_pointer].key_IDs[k] == key_list[q][p].key_IDs[rnd_id])
                            break;
                    }
                    if(k == M)
                    {
                        node_key_list[next_key_pointer].key_IDs[tmp_count] = key_list[q][p].key_IDs[rnd_id];
                        tmp_count++;
                    }
                }
                next_key_pointer++;
            }
        }        
    }           
}

/*------------------------------------------------------------------------------*
 * Following function count the no of nodes with a cluster head			*
 * use_kps 	- if 0 just count the no of nodes with a CH			*
 * 		- if 1 count the no of nodes with a CH due to unavailability	*
 * 		  of a common key with the CH					*
 * Retruns no of nodes with a CH						*
 *------------------------------------------------------------------------------*/
uint nodes_without_CH(uchar use_kps)
{
    uint x, y, no_ch, no_key;

    no_ch = 0;			//no of nodes with a CH
    no_key = 0;			//no of nodes with a CH due to lack of common key

    for (y = 0; y < NODESY ; y++)
    {
        for(x = 0 ; x < NODESX ; x++)
        {
                //node exist but not in a cluster
            if((nodes[x][y].NID != 0) && (nodes[x][y].CID == 0))
                no_ch++;

            //node exist but not in a cluster & due to lack of common key
            if((nodes[x][y].NID != 0) && (nodes[x][y].CID == 0) && (nodes[x][y].no_key_to_CH == 1))
                no_key++;
        }
    }

    if(use_kps == 1)		//no of nodes with a CH due to lack of common is key is needed
        return no_key;
    else				//else just return the no of nodes with a CH
        return no_ch;
}

/*------------------------------------------------------------------------------*
 * Following function print the no of common keys each node has with each & 	*
 * every node in its neighborhood to a file.					*
 *------------------------------------------------------------------------------*/
void print_no_common_keys()
{
    int minx, miny, maxx, maxy, common_keys;
    uint x0, y0, x, y, nid, z, k, l;
    float distance;
    key_index key_parent, key_child; 
   // FILE *com_keyfd;
    region my_region;
    char rec_rssi;
    uint com_key;
    
/*
    com_keyfd = fopen(COM_KEYFILE, "w");
    if(com_keyfd == NULL)
    {
            perror("ERROR: ");
            exit(1);
    }
*/

    for (y = 0; y < NODESY ; y++)
    {
        for(x = 0 ; x < NODESX ; x++)
        {		
            nid = nodes[x][y].NID;

            if(nid == 0)					//Skip if node doesn't exist
                continue;	
            x0 = (nid - 1) % NODESX;
            y0 = (nid - 1) / NODESX;

            my_region = get_node_region(x0, y0, R);		//get my region
            minx = my_region.minx;
            miny = my_region.miny;
            maxx = my_region.maxx;
            maxy = my_region.maxy;
                
            for(z = 0; z < M ; z++)
                key_parent.key_IDs[z] = node_key_list[nodes[x0][y0].key_pointer].key_IDs[z];

            for(l = miny; l <= maxy; l++)
            {
                for (k = minx; k <= maxx; k++)
                {
                    distance = sqrt((k - x0)*(k - x0)*(GRIDX * GRIDX) + (l - y0)*(l - y0)*(GRIDY * GRIDY));
                    rec_rssi = RSSI(distance, 0);     //determine the RSSI for the signal for the given distance
                    
                    // if within communication range & if the node exist & not the same node (distance != 0)
                    if ((rec_rssi >= 0) && (nodes[k][l].NID != 0) && (distance != 0)) 
                    {
                        for(z = 0; z < M ; z++)
                            key_child.key_IDs[z] = node_key_list[nodes[k][l].key_pointer].key_IDs[z];
                        
                        common_keys =  common_key(key_parent, key_child, &com_key);
                        if(common_keys != 0)
                        {
                            nodes[x0][y0].no_neighbors++;
                            nodes[x0][y0].no_key_neighbors++;
                        }
                        else
                            nodes[x0][y0].no_neighbors++;
                        
      //                  retvalue = sprintf(msg, "%d\n", common_keys);
      //                  fputs(msg, com_keyfd);
                    }
                }
            }
        }
    }
    //fclose(com_keyfd);
}
