#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "types.h"
#include "kps.h"

key_index *compromized_keys;	//hold the list of compromized nodes
uint no_compromized_keys = 0;			//no of compromized nodes

/*----------------------------------------------------------------------*
 * Follwoing function generate the last of key indexes and corrosponding*
 * key IDs.                                                             *
 * p - is the p of TD(k,p)                                              *
 * k - is the number of keys per block                                  *
 *----------------------------------------------------------------------*/
void generate_key_IDs(uint p, uint k, block *key_block)
{
	int i, j, x;

	for(i = 0 ; i < p; i++)			//for ordered pair (i, j), 0 <= i <= p-1 , 0 <= j <= p-1
	{
		for(j = 0; j < p; j++)
		{
			key_block[(i * p) + j].key_index[0] = i;	//key index
			key_block[(i * p) + j].key_index[1] = j;	//key index
			key_block[(i * p) + j].selected = 0;		//Key block not selected
			
			for(x = 0; x < k; x++)	//Each key in block, 0 <= x <= k -1 
			{
				key_block[(i * p) + j].key_id[2 * x] = x;	//key id A[i,j] = {x, (ix+j) mod p}  
				key_block[(i * p) + j].key_id[(2 * x) + 1] = (((i * x) + j) % p);
			}	
		}
	}
}

/*----------------------------------------------------------------------*
 * Follwoing function dump the key indexes and list of keys of a block  *
 * to a text file based on Lee-Stinson approach.                        *
 * File has the following format:                                       *
 * first raw - (v, b, r, k; z)                                          *
 * where v - total no of uniuqe keys, b - no of blocks, r - degree      *
 * k - rank & z - no of blocks to merge. In this case it should be 1    *
 * p - is the p of TD(k,p)                                              *
 * k - is the no of keys per block                                      *
 *----------------------------------------------------------------------*/
void dump_keys_ls(uint v, uint b, uint p, uint k, block *key_block)
{
	int i, retvalue, x;
	FILE *keyid_fd;
	char msg[200];
	
	keyid_fd = fopen(KEYFILE, "w");		//Open file for writing
	if(keyid_fd == NULL)				//If file can't be created
		perror("Error creating file:");
 
	//Dump (v, b, r, k; z) info to file
	retvalue = sprintf(msg, "%d\t%d\t%d\t%d\t%d\n", v, b, p, k, 1);
	fputs(msg, keyid_fd);                   //dump to the file
	
	for(i = 0; i < b; i++)				//Write all the key IDs b = p^2
	{
		//First print the node index
		retvalue = sprintf(msg, "%d\t%d\t", key_block[i].key_index[0], key_block[i].key_index[1]);
                fputs(msg, keyid_fd);			//dump to the file
		
		for(x = 0; x < k; x++)
		{
			//print individual key IDs
			retvalue = sprintf(msg, "%d\t%d\t", key_block[i].key_id[2 * x], key_block[i].key_id[(2 * x) + 1]);
                	fputs(msg, keyid_fd);		//dump to the file
		}
		fputs("\n", keyid_fd);

	}
	fclose(keyid_fd);				//Close file
}

/*----------------------------------------------------------------------*
 * Follwoing function dump the key indexes and list of keys of a block  *
 * to a text file based on Chakrabarti-Maitral-Roy approach.            *
 * File has the following format:                                       *
 * first raw - (v, b, r, k; z)                                          *
 * where v - total no of uniuqe keys, b - no of blocks, r - degree      *
 * k - rank & z - no of blocks to merge. In this case it should be > 1  *
 * p - is the p of TD(k,p)                                              *
 * k - is the no of keys per block                                      *
 * z - is the no of blocks to merge                                     *
 *----------------------------------------------------------------------*/
void dump_keys_cmr(uint v, uint b, uint p, uint k, uchar z, block *key_block)
{
        int i, j, retvalue, x, n, rnd_key_index, key_index[4];
        FILE *keyid_fd;
        char msg[1000];
        block *tmp_key_block;

        tmp_key_block = key_block;                      //Copy the pounter to a temporary pointer so that it can be incremented
	v = k * p;					//No of distinct keys
        b = p * p;					//Total no of blocks
	n = b / z ;					//Total no of nodes

	srand(time(NULL));                              //Set the seed for rand()

        keyid_fd = fopen(KEYFILE, "w");              //Open file for writing
        if(keyid_fd == NULL)                            //If file can't be created
                perror("Error creating file:");
	//Dump (v, b, r, k; z) info to file
        retvalue = sprintf(msg, "%d\t%d\t%d\t%d\t%d\n", v, b, p, k, z);
        fputs(msg, keyid_fd);                   	//dump to the file

        for(i = 0; i < n; i++)                          //Write all the key IDs for each node n
        {
		//First locate z blocks that are no used
                for(j = 0; j < z ; j++)			
		{
                	while(1)				//Loop forever
			{
				rnd_key_index = rand() % (b);	//Randomly select a block
				if(key_block[rnd_key_index].selected == 0)	//If found a key block that is not used
				{
					key_index[j] = rnd_key_index;
					key_block[rnd_key_index].selected = 1;		//Mark as already selected	
					break;						//Break the loop
				}
			}
		}
                for(j = 0; j < z ; j++)			//print the z many selected key index
		{
			retvalue = sprintf(msg, "%d\t%d\t", key_block[key_index[j]].key_index[0], key_block[key_index[j]].key_index[1]);
			fputs(msg, keyid_fd);                   //dump to the file
		}
		
                for(j = 0; j < z ; j++)			//print the individual key IDs on z many selected key indexes
		{
			for(x = 0; x < k; x++)
			{
				//print individual key IDs
				retvalue = sprintf(msg, "%d\t%d\t", key_block[key_index[j]].key_id[2 * x], key_block[key_index[j]].key_id[(2 * x) + 1]);
				fputs(msg, keyid_fd);           //dump to the file                
                	}               
		}
		fputs("\n", keyid_fd);
	}
		fclose(keyid_fd);                               //Close file
}

/*----------------------------------------------------------------------*
 * Following function determines a commom key given z number of key 	*
 * blocks from 2 sensor nodes.						*
 * Each block in node a will be compared against each node in node b	*
 * z 	-	No of key blocks in a node				*
 * a 	-	List of key indexes from node a				*
 * b 	-	List of key indexes from node b				*
 * k	-	No of keys in a key block				*
 * p 	- 	is the p of TD(k,p)                                     *
 * key_id -	Is the key id of the common key. If no common key	*
 * 		exist will be set to -1					*
 * Returns 	1 if common key exist, else 0				* 				
 *----------------------------------------------------------------------*/
int block_common_key(uchar z, key_index a[MAX_Z], key_index b[MAX_Z], uint k, uint p, key_index *key_id)
{
        key_index tmpa, tmpb;
	int n, m, result;

	if(z <= 0)
		return 0;                       //if no of blocks is <= 0 then no common key

	for(n = 0; n < z; n++)                  //Comapre each block in one node against all the pairs in other block
	{
		for(m = 0; m < z; m++)
		{
                        tmpa.a = a[n].a;		
                        tmpa.b = a[n].b;
                        tmpb.a = b[m].a;
                        tmpb.b = b[m].b;

                        result = common_key(tmpa, tmpb, k, p, key_id);	//Determine a common key for the given block

                        if(result == 1)		//if a common key is available
				return 1;	//stop when 1st common key is found
                }
	}
	return 0;				//if no common key is found return 0
}

/*----------------------------------------------------------------------*
 * Following function determines the number of commom keys given z 	*
 * number of key blocks from 2 sensor nodes.				*
 * Each block in node a will be compared against each node in node b	*
 * z 	-	No of key blocks in a node				*
 * a 	-	List of key indexes from node a				*
 * b 	-	List of key indexes from node b				*
 * k	-	No of keys in a key block				*
 * p 	- 	is the p of TD(k,p)                                     *
 * Returns 	no of common key, 0 if no common key exist		* 				
 *----------------------------------------------------------------------*/
int block_no_common_key(uchar z, key_index a[MAX_Z], key_index b[MAX_Z], uint k, uint p)
{
        key_index tmpa, tmpb, tmpc;
	int n, m, result;
	uchar no_keys = 0;

	if(z <= 0)
		return 0;                       //if no of blocks is <= 0 then no common key

	for(n = 0; n < z; n++)                  //Comapre each block in one node against all the pairs in other block
	{
		for(m = 0; m < z; m++)
		{
                        tmpa.a = a[n].a;		
                        tmpa.b = a[n].b;
                        tmpb.a = b[m].a;
                        tmpb.b = b[m].b;

                        result = common_key(tmpa, tmpb, k, p, &tmpc);	//Determine a common key for the given block

                        if(result == 1)		//if a common key is available
				no_keys++;	//increment the no of common keys
                }
	}
	return no_keys;				//if no common key is found return 0
}

/*----------------------------------------------------------------------*
 * Following function determines a commom key given 2 key blocks 	*
 * a 	-	List of key indexes from node a				*
 * b 	-	List of key indexes from node b				*
 * k	-	No of keys in a key block				*
 * p 	- 	is the p of TD(k,p)                                     *
 * key_id -	Is the key id of the common key. If no common key	*
 * key_id -	Is the key id of the common key. If no common key	*
 * 		exist will be set to -1					*
 * Returns 	1 if common key exist, else 0				* 				
 *----------------------------------------------------------------------*/
int common_key(key_index x, key_index y, uint k, uint p, key_index *key_id)
{
	int diff, i, j, inv, l;
	key_index tmp;

	if(x.a == y.a)				//if xa == xb no common key
		return 0;

	
	diff =  x.a - y.a;
	if(diff < 0)			//if diff is negative swap x & y
	{
		tmp = x;
		x = y;
		y = tmp;
	}

	inv = inv_mod(abs(diff), p);
	if(inv == 0)			//If no modular inverse no common key
		return 0; 
	i = emod(((y.b - x.b) * inv), p);
		
	if(( i >= 0) && (i <= (k-1)))	//if 0 <= x <= k -1 nodes share a key if x => k they don't share a key
	{
		j = emod((x.a * i + x.b), p);	// key id (i,j) = (x, (a1 * x + a2) mod p)
		key_id->a = i;
		key_id->b = j;

		//see whether the key ID is in the list of compromized IDs
		for(l = 0 ; l < no_compromized_keys; l++)
		{
			//if the key ID is in the list of compromized keys
			if((i == compromized_keys[l].a) && (j == compromized_keys[l].b)) 
			{
				return 0;
			}
		}
		return 1;		//Nodes share a common key
	}
	else 
		return 0;
}

/*------------------------------------------------------------------------------*
 * Following function defines the Euclidena mod operation. Whihc works for both	*
 * a < 0 and n < 0. If both a & n are positive (given n != 0) then its the 	*
 * standard mod operation. If n == 0 mod operation is not defined		*
 * Refer paper titled "The Euclidean Definition of the Functions div and mod" 	*
 * for the definition of emod							*
 * a 	- 	dividend							*
 * n	-	divisor								*
 * return the modulas. If unable to determine emod -1 is returned. if correct	*
 * emod >= 0									*
 *------------------------------------------------------------------------------*/
int emod(int a, int n)
{
	int result = 0;
	unsigned char i = 0;

	if( n == 0)
		return -1;
	if ((a >= 0) && (n > 0))	//if both a >= 0 & n > 0 it is the standard mod
		return (a % n);			
	
	else if((n < 0) && (a >= 0))		//since; a mod (-n) = a mod n
		result = a % abs(n);	
	else if((n > 0) && (a < 0))
	{
		if((abs(a) % n) == 0)	//if a/n is a integer i = 0 (a devidable by n) else i = 1
			i = 0;
		else 
			i = 1;
		result = -1 * (abs(a) % n) + n * i; // -a mod n = -( a mod n) + nij, j is the signe of n
	}
	else if((n < 0) && (a < 0))
	{
		if((abs(a) % n) == 0)	//if a/n is a integer i = 0 (a devidable by n) else i = 1
			i = 0;
		else 
			i = 1;
		result = -1 * (abs(a) % abs(n)) + -1 * n * i; // -a mod (-n) = -(a mod (-n)) + nij = -(a mod n) + nij, j is the sign of n
	}
	return result;			//return emod
}

/*----------------------------------------------------------------------*
 * Follwoing function generate the Extended Greatest Common Divisor used*
 * by the inv_mod function to calcualte the inverse modulus		*
 * Extended form of Euclid's algorithm computes g = GCD(a, b) of 2 	*
 * integers a & b and also determines u & v such that 			*
 * g = GCD(a, b) = a*u + b*v.						*
 * a 	- 	1st number						*
 * b	- 	2nd number						*
 * return (g, u, v)							*
 *----------------------------------------------------------------------*/
guv ext_gcd(int a, int b)
{
	int	u, v, g, u1, v1, g1, q, t1, t2, t3;
	guv tmp_guv;

	a = abs(a);			//Conver to |a| & |b|
	b = abs(b);

	u = 1;				//Set intial parameters
	v = 0;
	g = a;
	u1 = 0;
	v1 = 1;
	g1 = b;

	while(g1 != 0)
	{
		q = g/g1;		//Integer divison
		t1 = u - q*u1; 
		t2 = v - q*v1; 
		t3 = g - q*g1; 
		u = u1; 
		v = v1; 
		g = g1; 
		u1 = t1; 
		v1 = t2; 
		g1 = t3;
	}
	tmp_guv.g = g;
	tmp_guv.u = u;
	tmp_guv.v = v;
	return tmp_guv;
}

/*----------------------------------------------------------------------*
 * Follwoing function Solve linear congruence equation 			*
 * a * z == 1 (mod p)							*
 * z is the modular inverse of a (mod n). To solve for z given x & p 	*
 * uses the extended form of Euclid's algorithm				*
 * Returns inverse modulus of a. If no inverse exist 0 is returned	*
 *----------------------------------------------------------------------*/
int inv_mod(int a, int p)
{ 
	guv tmp_guv;

	p = abs(p); 
	a = emod(a, p); 			// % is the remainder function, 0 <= x % n < |n| 
	tmp_guv = ext_gcd(a, p); 

	if (tmp_guv.g != 1)
		return 0;		//No inverse 
	else 
		return (emod(tmp_guv.u, p)); 
}

uchar add_compromized_blocks(key_index block[MAX_Z], uchar p, uchar k, uchar z)
{
	int l, x;
	uchar i, j;

	if(compromized_keys == NULL)		//if memory is not already allocated
		compromized_keys = (key_index *)malloc(sizeof(key_index) * k * MAX_Z * COMP_NODES);	//hold the list of compromized nodes
	
	if(compromized_keys == NULL)		//if unable to allocate memory
	{
		printf("Unable to allocate memeory, add_compromized_blocks(). Terminating....\n");
		exit(1);
	}

	if(no_compromized_keys == (k * MAX_Z * COMP_NODES))	//If no more nodes can be added
		return 0;
	else						//if enough space is available add
	{
		for(l = 0 ; l < z; l++)		//For each key block
		{
			i = block[l].a;
			j = block[l].b;

			for(x = 0 ; x < k ; x++)	//for 0 <= x <= k -1 
			{
				compromized_keys[(no_compromized_keys + (l * k) + x)].a = x;
				compromized_keys[(no_compromized_keys + (l * k) + x)].b = (((i * x) + j) % p);
			}	
		}
		no_compromized_keys += z * k;	//Increment the counter
		return 1;
	}
}	
