#include "energy.h"
#include <math.h>
#include <stdio.h>


/*----------------------------------------------------------------------*
 * Following function calculate the amount of energy required to receive*
 * a data packet of a geven length					*
 * no_bits - no of bits							*
 *----------------------------------------------------------------------*/
float energy_to_receive(int no_bits) 
{
    return (float)(no_bits * E_CIRCUIT);
}

/*----------------------------------------------------------------------*
 * Following function calculate the amount of energy required to 	*
 * transmit a data packet of given size and to a given distance		*
 * If the distance is less than R * CH_CH_R_FACT then free space model	*
 * is used. Otherwice multi=path fading model is used			*
 * no_bits - no of bits to transmit					*
 * distace - distance between communicationg nodes or maximum 		*
 * transmission range 							*
 * fs_or_mp - Fress space model or multi-path model to use		*
 * if fs_or_mp = 0 free space						*
 * if fs_or_mp = 1 multi-path						*
 *----------------------------------------------------------------------*/
float energy_to_transmit(int no_bits, float distance) 
{
    float circuit_energy, amp_energy;
    
    circuit_energy = (float)(no_bits * E_CIRCUIT);
    
    if(distance < T_RANGE)						//If below threshold use free space model
        amp_energy = no_bits * E_AMP_FS * distance * distance;
    else								//Else use multipath fading model
        amp_energy = no_bits * E_AMP_MP * pow(distance, 4);
    
    return (amp_energy + circuit_energy);
}

/*----------------------------------------------------------------------*
 * Function calculates Received Signal Strength Indicator (RSSI) value	*
 * based on the equation						*
 *	P_r(d) = 10log(kP_t) - 10nlog(d)				*
 * where P_r is the receiver power in dBm, D_0 is the close-in distance *
 * n is the path loss exponent						*
 * distance - distance between tranmisster & receiver                   *
 * lp       - whether to use LP or high power
 *----------------------------------------------------------------------*/
char RSSI(float distance, unsigned char lp)
{
    double trans_power, rec_power, diff;

    if(lp == 0)
        trans_power = pow(10, (RF_POWER_LP/10)); //convert to power in mW
    else
        trans_power = pow(10, (RF_POWER_HP/10)); //convert to power in mW

    rec_power = 10 * log10(MY_K * trans_power) - 10 * MY_N * log10(distance);
    diff = fabs(REC_SENSITIVITY) - fabs(rec_power);	
    if(diff < 0)
        diff = diff - 1;

    return (char)diff;
}	

/*----------------------------------------------------------------------*
 * Detemine the transmission range                                      *
 * lp - If 0 use low power else use High power                          * 
 *----------------------------------------------------------------------*/
float transmission_range(unsigned char lp)
{
    double trans_power, distance;
    
    if(lp == 0)                 //Use Low Power
    {
        trans_power = pow(10, (RF_POWER_LP/10)); //convert to power in mW
        distance = pow(10, (10*log10(MY_K * trans_power) - REC_SENSITIVITY)/(10 * MY_N));   
    }
    else                        //Use High Power
    {
        trans_power = pow(10, (RF_POWER_HP/10)); //convert to power in mW
        distance = pow(10, (10*log10(MY_K * trans_power) - REC_SENSITIVITY)/(10 * MY_N));   
    }
    return (float)distance;
}
