#include "types.h"
#include <stdio.h>
#include <stdlib.h>

/*------------------------------------------------------------------------------*
 * following function generates the hierarchical cluster id based on the parent	*
 * cluster id & node depth. cids are represented in the following manner	*
 *     +------------------------+----+------------------+			*
 *     |  ......|child2|child1|parent|  depth (5- bits)	|			*
 *     +------------------------+----+------------------+			*
 *     depth - 5 bits - can represent upto depth of 32 levels in the tree 	*
 *     branching factor (b) - up to 8 bracnhes (3 bits to represent each brancg	*
 *     number. 									*
 *     child id = parent id + (branch no << ((depth -1) * b)+ 5) + depth 	*
 *     with 96-bit cid up to 30 level cluster tree can be represented		*
 *     however there are no 64-bit integer data structures. so an array of two	*
 *     32-bit integers are used.						*  
 *------------------------------------------------------------------------------*/
Hie_CID generate_CID(Hie_CID parent_id, uint child_no, uchar depth)
{
	int tmp_id0, tmp_id1, tmp_id2, tmp_parent_id[3], tmp_depth;
	Hie_CID new_h_cid;
	
	tmp_parent_id[0] = parent_id.id[0];
	tmp_parent_id[1] = parent_id.id[1];
	tmp_parent_id[2] = parent_id.id[2];
	
	tmp_depth = tmp_parent_id[2] & 31;			//extract the depth 31 = 11111 in binary

	if ((tmp_depth + 1) != depth)				//if child depth != parent depth + 1
	{
		printf("child depth should be equal to parent depth + 1. Child: %d Parent: %d\n", depth, tmp_depth);
		exit(0);
	}
	
	if(depth <= 9)					//up to depth of 9 can be represented by 32-bits (3 * 9 + 5)
	{
		tmp_id2 = child_no;
		tmp_id2 = tmp_id2 << (((depth -1) * 3) + 5); 
		tmp_id2 += tmp_parent_id[2];
		tmp_id2++;				//increment depth by 1 
		tmp_id1 = 0;				//the the last (msb) 32-bits zero
		tmp_id0 = 0;				//the the last (msb) 32-bits zero
	}
	else if((depth > 9) && (depth < 20))                                    //If within 10 to 19
        {
                tmp_id2 = tmp_parent_id[2];
                tmp_id2++;                              //Increment depth
                tmp_id1 = child_no;                     //Move to the other part of the CID and set it
                tmp_id1 = tmp_id1 << ((depth - 10) * 3);//Shift the new no
                tmp_id1 += tmp_parent_id[1];            //Add the MSB part of parent
                tmp_id0 = 0;                            //The the last (MSB) 32-bits zero
        }
	else                                            //if not
        {
                tmp_id2 = tmp_parent_id[2];
                tmp_id2++;                              //increment depth
                tmp_id1 = tmp_parent_id[1];
                tmp_id0 = child_no;                     //move to the other part of the cid and set it
                tmp_id0 = tmp_id0 << ((depth - 20) * 3);//shift the new no
                tmp_id0 += tmp_parent_id[0];            //add the msb part of parent
        }

        new_h_cid.id[0] = tmp_id0;
        new_h_cid.id[1] = tmp_id1;
        new_h_cid.id[2] = tmp_id2;
        return new_h_cid;                                       //return new cid
}

/*------------------------------------------------------------------------------*	
 * Following function determines when the next new cluster event should start	*
 * Starting time is determined based on breadth-first tree formation		*
 * Each event in the next round should be scheduled after the completion of the *
 * last event in the corrent round						*
 * depth	-	Depth of the event in the next round			*
 * no_CCHs	- 	no of candidate CHs					*
 * cluster_time	- 	time to form a cluster (timeout)			*
 * CCH_delay	-	Delay for the next child CH for the same parent CH	*
 * returns 	- 	time of the last event					*
 *------------------------------------------------------------------------------*/ 
uint last_event_time(uchar depth, uchar no_CCHs, uint cluster_time, uint CCH_delay)
{
	//dt_CH + (no_CCHs -1)(d -1)t_CCH
	return (depth * cluster_time + (no_CCHs - 1) * (depth -1) * CCH_delay);

}

/*------------------------------------------------------------------------------*
 * Following function return the range of a node given its communication range	*
 * x0		- x coordinate of node						*
 * y0		- y coordinate of node						*
 * r	 	- cimmunication range of node					*
 * region	- x & y coordinates of the rectangle				*
 *------------------------------------------------------------------------------*/
region get_node_region(uint x0, uint y0, float r)
{
	region tmp_region;
	int minx, maxx, miny, maxy;
	
	minx = x0 - (r/GRIDX) - 1;                              // 1 hop neighbors are in the range of (x0 - r) to (x0 + r)
	if (minx < 0)                                           // stay within the grid			
		minx = 0;
	maxx = x0 + (r/GRIDX) + 1;
	if(maxx >= NODESX)
		maxx = NODESX -1;

	miny = y0 - (r/GRIDY) - 1;                              // 1 hope neighbors are in the range of (y0 - r) tp (y0 + r)
	if (miny < 0)                                           // stay within the grid
		miny = 0;		
	maxy = y0 + (r/GRIDY) + 1;
	if(maxy >= NODESY)
		maxy = NODESY - 1;

	tmp_region.minx = minx;
	tmp_region.maxx = maxx;
	tmp_region.miny = miny;
	tmp_region.maxy = maxy;

	return tmp_region;
}

/*------------------------------------------------------------------------------*
 * Follwoing function generates a random time to wait before taking some action	*
 * return	- 	+/- random time						*
 *------------------------------------------------------------------------------*/
char random_wait_time()
{
	// +/-random(r)
	return (rand() % MAX_RND_TIME) - BIAS_POINT;
}

/*------------------------------------------------------------------------------*
 * Follwoing function maps the Cluster ID to the corrosponding symbol that 	*
 * appears on screen								*
 *------------------------------------------------------------------------------*/
char CID_to_symbol_mapping(int CID)
{	
	char symbol;
	switch (CID)
	{
		case 0:
			symbol = 'o' ;
			break;
		case 1:	
			symbol = '1' ;
			break;
		case 2:
			symbol = '2' ;
			break;
		case 3:
			symbol = '3' ;
			break;
		case 4:
			symbol = '4' ;
			break;
		case 5:
			symbol = '5' ;
			break;
		case 6:
			symbol = '6' ;
			break;
		case 7:
			symbol = '7' ;
			break;
		case 8: 
			symbol = '8' ;
			break;
		case 9:
			symbol = '9' ;
			break; 
		case 10:
			symbol = 'A' ;
			break;
		case 11:
			symbol = 'B' ;
			break;
		case 12:
			symbol = 'C' ;
			break;
		case 13:
			symbol = 'D' ;
			break;
		case 14:
			symbol = 'E' ;
			break;
		case 15:
			symbol = 'F' ;
			break;
		case 16:
			symbol = 'G' ;
			break;
		case 17: 
			symbol = 'H' ;
			break;
		case 18: 
			symbol = 'I' ;
			break;
		case 19:
			symbol = 'J' ;
			break;
		case 20: 
			symbol = 'K' ;
			break;
		case 21: 
			symbol = 'L' ;
			break;
		case 22: 
			symbol = 'M' ;
			break;
		case 23:
			symbol = 'N' ;
			break;
		case 24:
			symbol = 'O' ;
			break;
		case 25:
			symbol = 'P' ;
			break;
		case 26:
			symbol = 'Q' ;
			break;
		case 27:
			symbol = 'R' ;
			break;
		case 28:
			symbol = 'S' ;
			break;	
		case 29:
			symbol = 'T' ;
			break;
		case 30:
			symbol = 'U' ;
			break;
		case 31:
			symbol = 'V' ;
			break;
		case 32:
			symbol = 'W' ;
			break;
		case 33:
			symbol = 'X' ;
			break;
		case 34:	
			symbol = 'Y' ;
			break;
		case 35: 
			symbol = 'Z' ;
			break;
		case 36:
			symbol = '*' ;
			break;
		case 37:
			symbol = '#' ;
			break;
		case 38:
			symbol = '$' ;
			break;
		case 39:
			symbol = '@' ;
			break;
		case 40:
			symbol = '+' ;
			break;
		case 41:
			symbol = '-' ;
			break;
		case 42:
			symbol = '&' ;
			break;
		case 43:
			symbol = '<' ;
			break;
		case 44:
			symbol = '!' ;
			break;
		case 45:
			symbol = '=' ;
			break;
		case 46:
			symbol = 'a' ;
			break;
		case 47:
			symbol = 'b' ;
			break;
		case 48:
			symbol = 'c' ;
			break;
		case 49:
			symbol = 'd' ;
			break;
		case 50:
			symbol = 'e' ;
			break;
		case 51:
			symbol = 'f' ;
			break;
		case 52:
			symbol = 'g' ;
			break;
		case 53:
			symbol = 'h' ;
			break;
		case 54:
			symbol = 'i' ;
			break;
		case 55:
			symbol = 'j' ;
			break;
		case 56:
			symbol = 'k' ;
			break;
		case 57:
			symbol = 'l' ;
			break;
		case 58:
			symbol = 'm' ;
			break;
		case 59:
			symbol = 'n' ;
			break;
		case 60:
			symbol = 'p' ;
			break;
		case 61:
			symbol = 'q' ;
			break;
		case 62:
			symbol = 'r' ;
			break;
		case 63:
			symbol = 's' ;
			break;
		case 64:
			symbol = 't' ;
			break;
		case 65:
			symbol = 'u' ;
			break;
		case 66:
			symbol = 'v' ;
			break;
		case 67:
			symbol = 'w' ;
			break;
		case 68:
			symbol = 'x' ;
			break;
		case 69:
			symbol = 'y' ;
			break;
		case 70:
			symbol = 'z' ;
			break;
		default:
			symbol = '?' ;
		break;
	}
	return symbol;
}

