#include "types.h"
#include <stdio.h>
#include <stdlib.h>

/*------------------------------------------------------------------------------*
 * following function generates the hierarchical cluster id based on the parent	*
 * cluster id & node depth. cids are represented in the following manner	*
 *     +------------------------+----+------------------+			*
 *     |  ......|child2|child1|parent|  depth (6- bits)	|			*
 *     +------------------------+----+------------------+			*
 *     depth - 6 bits - can represent upto depth of 38 levels in the tree 	*
 *     branching factor (b) - up to 8 bracnhes (3 bits to represent each brancg	*
 *     number. 									*
 *     child id = parent id + (branch no << ((depth -1) * b)+ 6) + depth 	*
 *     with 128-bit CID up to 38 level cluster tree can be represented		*
 *     however there are no 128-bit integer data structures. so an array of four*
 *     32-bit integers are used.						*
 *------------------------------------------------------------------------------*/
Hie_CID generate_CID(Hie_CID parent_id, uint child_no, uchar depth) 
{
    int tmp_id0, tmp_id1, tmp_id2, tmp_id3, tmp_parent_id[4], tmp_depth;
    Hie_CID new_h_cid;

    tmp_parent_id[0] = parent_id.id[0];
    tmp_parent_id[1] = parent_id.id[1];
    tmp_parent_id[2] = parent_id.id[2];
    tmp_parent_id[3] = parent_id.id[3];

    tmp_depth = tmp_parent_id[3] & 63;	//extract the depth 64 = 111111 in binary

    if ((tmp_depth + 1) != depth) //if child depth != parent depth + 1
    {
        printf("child depth should be equal to parent depth + 1. Child: %d Parent: %d\n", depth, tmp_depth);
        exit(0);
    }
    if(depth > 38)
    {
        printf("Hierarchical address overflow.\n");
        exit(1);
    }

    if(depth <= 8)	//up to depth of 8 can be represented by 32-bits (3 * 8 + 6)
    {
        tmp_id3 = child_no;
        tmp_id3 = tmp_id3 << (((depth -1) * 3) + 6); 
        tmp_id3 += tmp_parent_id[3];
        tmp_id3++;			//increment depth by 1 
        tmp_id2 = 0;			//the the last (msb) 32-bits zero
        tmp_id1 = 0;			//the the last (msb) 32-bits zero
        tmp_id0 = 0;			//the the last (msb) 32-bits zero
    }
    else if((depth > 8) && (depth < 19))    //If within 9 to 18
    {
        tmp_id3 = tmp_parent_id[3];
        tmp_id3++;                      //Increment depth
        tmp_id2 = child_no;             //Move to the other part of the CID and set it
        tmp_id2 = tmp_id2 << ((depth - 9) * 3);//Shift the new no
        tmp_id2 += tmp_parent_id[2];    //Add the MSB9 part of parent
        tmp_id1 = 0;                    //The the last (MSB) 32-bits zero
        tmp_id0 = 0;                    //The the last (MSB) 32-bits zero
    }
    else if((depth > 18) && (depth < 29))   //If within 19 to 28                                        
    {
        tmp_id3 = tmp_parent_id[3];
        tmp_id3++;                      //increment depth
        tmp_id2 = tmp_parent_id[2];
        tmp_id1 = child_no;             //move to the other part of the cid and set it
        tmp_id1 = tmp_id1 << ((depth - 19) * 3);//shift the new no
        tmp_id1 += tmp_parent_id[1];    //add the msb part of parent
        tmp_id0 = 0;                    //add the msb part of parent
    }
    else                                //If within 29 to 38
    {
        tmp_id3 = tmp_parent_id[3];
        tmp_id3++;                      //increment depth
        tmp_id2 = tmp_parent_id[2];
        tmp_id1 = tmp_parent_id[1];
        tmp_id0 = child_no;             //move to the other part of the cid and set it
        tmp_id0 = tmp_id0 << ((depth - 29) * 3);//shift the new no
        tmp_id0 += tmp_parent_id[0];    //add the msb part of parent
    }    

    new_h_cid.id[0] = tmp_id0;
    new_h_cid.id[1] = tmp_id1;
    new_h_cid.id[2] = tmp_id2;
    new_h_cid.id[3] = tmp_id3;
    return new_h_cid;                   //return new cid
}



uchar hop_distance(Hie_CID source_add, Hie_CID dest_add)
{
    uchar source_depth, dest_depth, min_depth, tmp_source_add, tmp_dest_add, i, hops;
    
    source_depth = source_add.id[3] & 63; //63 = 111111 in binary
    dest_depth = dest_add.id[3] & 63;

    if((source_depth > 38) || (dest_depth > 38))    //If depth > 38 Hierarchical Address overflow
            return 0;

    if(source_depth > dest_depth)	//Find minimum depth 
        min_depth = dest_depth;
    else
        min_depth = source_depth;

    for(i = 0; i < min_depth ; i++)
    {
        if(i < 8)	//If min-depth is within depth of 8. Check only the LSB (32-bits) of the address
        {	
            //Remove depth & hierarchical address of parent CHs starting from root node
            tmp_source_add = (source_add.id[3] >> (6 + i * 3)); 
            tmp_dest_add = (dest_add.id[3] >> (6 + i * 3));
            tmp_source_add = tmp_source_add & 7;  // 7 = 111 in binary. Extract only branch number
            tmp_dest_add = tmp_dest_add & 7;
            if(tmp_source_add != tmp_dest_add)	//If a none matching branch is found
                break;
        }
        else if((i >= 8) && (i < 18))	//Check the next 32-bits
        {
            //Remove the depth info from address 
            tmp_source_add = (source_add.id[2] >> ((i - 8) * 3));	
            tmp_dest_add = (dest_add.id[2] >> ((i - 8) * 3));
            tmp_source_add = tmp_source_add & 7;    // 7 = 111 in binary. Extract only branch number
            tmp_dest_add = tmp_dest_add & 7;	
            if(tmp_source_add != tmp_dest_add)	//If a none matching branch is found	
                break;
        }
        else if((i >= 18) && (i < 28))	//Check the next 32-bits
        {
            //Remove the depth info from address 
            tmp_source_add = (source_add.id[1] >> ((i - 18) * 3));	
            tmp_dest_add = (dest_add.id[1] >> ((i - 18) * 3));
            tmp_source_add = tmp_source_add & 7;    // 7 = 111 in binary. Extract only branch number
            tmp_dest_add = tmp_dest_add & 7;
            if(tmp_source_add != tmp_dest_add)	//If a none matching branch is found	
                break;
        }
        else
        {
            //Remove the depth info from address 
            tmp_source_add = (source_add.id[0] >> ((i - 28) * 3));
            tmp_dest_add = (dest_add.id[0] >> ((i - 28) * 3));
            tmp_source_add = tmp_source_add & 7;    // 7 = 111 in binary. Extract only branch number
            tmp_dest_add = tmp_dest_add & 7;
            if(tmp_source_add != tmp_dest_add)	//If a none matching branch is found	
                break;
        }               
    }

    //Distance to common branching point
    hops = (source_depth - i) + (dest_depth - i);	
    return hops;
}
/*------------------------------------------------------------------------------*
 * Following function determines when the next new cluster event should start	*
 * Starting time is determined based on breadth-first tree formation		*
 * Each event in the next round should be scheduled after the completion of the *
 * last event in the corrent round						*
 * depth	-	Depth of the event in the next round			*
 * no_CCHs	- 	no of candidate CHs					*
 * cluster_time	- 	time to form a cluster (timeout)			*
 * CCH_delay	-	Delay for the next child CH for the same parent CH	*
 * returns 	- 	time of the last event					*
 *------------------------------------------------------------------------------*/
uint last_event_time(uchar depth, uchar no_CCHs, uint cluster_time, uint CCH_delay) 
{
    //dt_CH + (no_CCHs -1)(d -1)t_CCH
    return (depth * cluster_time + (no_CCHs - 1) * (depth -1) * CCH_delay);
    
}


/*------------------------------------------------------------------------------*
 * Following function return the range of a node given its communication range	*
 * x		- x coordinate of node						*
 * y		- y coordinate of node						*
 * r	 	- cimmunication range of node					*
 * region	- x & y coordinates of the rectangle				*
 *------------------------------------------------------------------------------*/
region get_node_region(uint x, uint y, float r)
{
    region tmp_region;
    int minx, maxx, miny, maxy;
    
    minx = x - (r/GRIDX) - 1;    // 1 hop neighbors are in the range of (x - r) to (x + r)
    if (minx < 0)                 // stay within the grid
        minx = 0;
    maxx = x + (r/GRIDX) + 1;
    if(maxx >= NODESX)
        maxx = NODESX -1;
    
    miny = y - (r/GRIDY) - 1;    // 1 hope neighbors are in the range of (y - r) tp (y + r)
    if (miny < 0)                 // stay within the grid
        miny = 0;
    maxy = y + (r/GRIDY) + 1;
    if(maxy >= NODESY)
        maxy = NODESY - 1;
    
    tmp_region.minx = minx;     //set bounds
    tmp_region.maxx = maxx;
    tmp_region.miny = miny;
    tmp_region.maxy = maxy;
    
    return tmp_region;
}

/*------------------------------------------------------------------------------*
 * Follwoing function generates a random time to wait before taking some action	*
 * return	- 	+/- random time						*
 *------------------------------------------------------------------------------*/
char random_wait_time() 
{
    return (rand() % MAX_RND_TIME) - BIAS_POINT; // +/-random(r)
}

/*------------------------------------------------------------------------------*
 * Follwoing function maps the Cluster ID to the corrosponding symbol that 	*
 * appears on screen								*
 *------------------------------------------------------------------------------*/
char CID_to_symbol_mapping(int CID) 
{
    char symbol;
    switch (CID) 
    {
        case 0:
            symbol = 'o' ;
            break;
        case 1:
            symbol = '1' ;
            break;
        case 2:
            symbol = '2' ;
            break;
        case 3:
            symbol = '3' ;
            break;
        case 4:
            symbol = '4' ;
            break;
        case 5:
            symbol = '5' ;
            break;
        case 6:
            symbol = '6' ;
            break;
        case 7:
            symbol = '7' ;
            break;
        case 8:
            symbol = '8' ;
            break;
        case 9:
            symbol = '9' ;
            break;
        case 10:
            symbol = 'A' ;
            break;
        case 11:
            symbol = 'B' ;
            break;
        case 12:
            symbol = 'C' ;
            break;
        case 13:
            symbol = 'D' ;
            break;
        case 14:
            symbol = 'E' ;
            break;
        case 15:
            symbol = 'F' ;
            break;
        case 16:
            symbol = 'G' ;
            break;
        case 17:
            symbol = 'H' ;
            break;
        case 18:
            symbol = 'I' ;
            break;
        case 19:
            symbol = 'J' ;
            break;
        case 20:
            symbol = 'K' ;
            break;
        case 21:
            symbol = 'L' ;
            break;
        case 22:
            symbol = 'M' ;
            break;
        case 23:
            symbol = 'N' ;
            break;
        case 24:
            symbol = 'O' ;
            break;
        case 25:
            symbol = 'P' ;
            break;
        case 26:
            symbol = 'Q' ;
            break;
        case 27:
            symbol = 'R' ;
            break;
        case 28:
            symbol = 'S' ;
            break;
        case 29:
            symbol = 'T' ;
            break;
        case 30:
            symbol = 'U' ;
            break;
        case 31:
            symbol = 'V' ;
            break;
        case 32:
            symbol = 'W' ;
            break;
        case 33:
            symbol = 'X' ;
            break;
        case 34:
            symbol = 'Y' ;
            break;
        case 35:
            symbol = 'Z' ;
            break;
        case 36:
            symbol = '*' ;
            break;
        case 37:
            symbol = '#' ;
            break;
        case 38:
            symbol = '$' ;
            break;
        case 39:
            symbol = '@' ;
            break;
        case 40:
            symbol = '+' ;
            break;
        case 41:
            symbol = '-' ;
            break;
        case 42:
            symbol = '&' ;
            break;
        case 43:
            symbol = '<' ;
            break;
        case 44:
            symbol = '!' ;
            break;
        case 45:
            symbol = '=' ;
            break;
        case 46:
            symbol = 'a' ;
            break;
        case 47:
            symbol = 'b' ;
            break;
        case 48:
            symbol = 'c' ;
            break;
        case 49:
            symbol = 'd' ;
            break;
        case 50:
            symbol = 'e' ;
            break;
        case 51:
            symbol = 'f' ;
            break;
        case 52:
            symbol = 'g' ;
            break;
        case 53:
            symbol = 'h' ;
            break;
        case 54:
            symbol = 'i' ;
            break;
        case 55:
            symbol = 'j' ;
            break;
        case 56:
            symbol = 'k' ;
            break;
        case 57:
            symbol = 'l' ;
            break;
        case 58:
            symbol = 'm' ;
            break;
        case 59:
            symbol = 'n' ;
            break;
        case 60:
            symbol = 'p' ;
            break;
        case 61:
            symbol = 'q' ;
            break;
        case 62:
            symbol = 'r' ;
            break;
        case 63:
            symbol = 's' ;
            break;
        case 64:
            symbol = 't' ;
            break;
        case 65:
            symbol = 'u' ;
            break;
        case 66:
            symbol = 'v' ;
            break;
        case 67:
            symbol = 'w' ;
            break;
        case 68:
            symbol = 'x' ;
            break;
        case 69:
            symbol = 'y' ;
            break;
        case 70:
            symbol = 'z' ;
            break;
        default:
            symbol = '?' ;
            break;
    }
    return symbol;
}

/*------------------------------------------------------------------------------*
 * Following function sort the neighbor routing entries in the increasing order	*
 * based on hop count. If 2 neighbors has the same hop count they are swapped	*
 * by tossing a coin. This prevents the same neighbor being slected again & 	*
 * again when they have the same hop count - allow load to be disteributed	*
 * neigh - pointer to the neighbor routing array				* 
 * n - no of entries in the neighbor array                                      *
 *------------------------------------------------------------------------------*/
void bubble_sort(nei_status *neigh, int n)
{
    char swapped;
    int i;
    nei_status tmp;

    do
    {
        swapped = 0;
        for(i = 0; i < (n-1); i++)
        {
            if(neigh[i].hops > neigh[i+1].hops)
            {
                swapped = 1;
                tmp = neigh[i+1];
                neigh[i+1] = neigh[i];
                neigh[i] = tmp;
            }
            else if(neigh[i].hops == neigh[i+1].hops)
            {
                //Toss a coin & decide wich entry to use if 0 change if 1 keep  previous entry
                if((rand() % 2) == 0 ) 
                {
                    tmp = neigh[i+1];
                    neigh[i+1] = neigh[i];
                    neigh[i] = tmp;
                }
            }
        }
    }
    while (swapped == 1);   //Repeat until no swapping can be done.
}
