/*------------------------------------------------------------------------------*
 * Version - 8.2                                                                *
 * By - Dilum Bandara                                                           *
 * e-mail - dilumb@engr.colostate.edu                                           *
 * This implementation supports the following functions                         *
 * 	Basic HHC algorithm                                                     *
 * 	Breadth First tree formation                                            *
 * 	Hierarchical naming                                                     *
 * 	Consider node collisions                                                *
 *      Noise and RSSI                                                          *
 * 	CCHs send ACK only if they don't hear an ACK from a neighboring node    *
 *      Node energy consumption - send/receive                                  *
 *      Cluster and Tree optimization phase                                     *
 *      Circularity calculation                                                 *
 *      Routing                                                                 *
 *	VSN formation and inter-VSN communication				*
 *------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include "simulator.h"
#include "energy.h"

node nodes[NODESX][NODESY];		//Hold the node information
event *root;				//Root of the CH event list
char msg[100];                          //Hold temporary messages before dumpting data
uint next_cid = 1;                      //Next CID
uint collision_nodes[NO_COLLISION_NODES];	//List of nodes in the collision region for current time
uint no_collision_nodes = 0;		//Number of nodes in the collision region
uint last_collision_set[2] = {0, 0};	//Last 2 nodes that caused the collision
uint last_type3_event_time = 0;         //Time related to forming my own cluster
int event_nodes[NO_EVENT_NODES];        //Hold the list of nodes related to the event 
uchar optimized = 0;                    // Node optimization is not done
float R;                                //Tranmission range based on power & fading factor
int no_sin_hop = 0;
int no_mulhop = 0;

int main() 
{ 
    init(USE_NODE_FILE);		//initialize nodes & set relevent parameters. Set the 1st event (from root node)
    while(process_event_list()); 	//Process the event list until no events are found    
    
    //enable/disable following functions based on the properties that needs to be measured 
    if(optimized == 0)  //If not optimization phase is already run
    {
        //calculate_circularity(1);       //Dump node circularity info to file
        //print_nodes(SHOW_NODE_DATA, 1); //Dump node information to file
        //print_cluster_energy();         //Dump node energy info to file        
        optimized = 1;                  //Set as optimized
/*
        opti_none_cluster_nodes();      //Optimize none cluster nodes        
        form_my_own_cluster();          //If can't join a cluster form my own
        while(process_event_list()); 	//Process the event list until no events are found
        //Optimze cluster tree          
        opti_cluster_tree(nodes[STARTX][STARTY].NID, nodes[STARTX][STARTY].NID, 0, 0, MAX_TTL);
        while(process_event_list()); 	//Process the event list until no events are found
        update_child_nodes();           //Update child nodes depth info
 */       
    }
        
    //calculate_circularity(2);	//Dump node circularity info to file
    print_nodes(SHOW_NODE_DATA, 1); //Dump node information to file
    //inform_neighbors();    
    //discover_neighbors_of_link(8);
    //send_data();                  //Send data until packet drops        
    form_vsn();
    //send_vsn_unicast_data();
    //send_vsn_multicast_data();
    //print_nodes(SHOW_NODE_DATA, 1); //Dump node information to file
    //print_cluster_energy();       //Dump node energy info to file
    //form_second_cluster_tree(3);  //Form a 2nd cluster tree
    //who_died(4);
    gaussian_rnd_remove();          //Free memory allocated for random no generator
    exit(0);
}


/*------------------------------------------------------------------------------*
 * Following function initialise the nodes & the event list                     *
 * Nodes can be either randomly genrated or assigned based on a pregenerated 	*
 * file. 									*
 * use_file - if 1 use the pregenerated node id file, else place nodes randomly	*
 *------------------------------------------------------------------------------*/
void init(uchar use_file) 
{
    int count, x, y, i, j, start_NID, NID;
    FILE *nodes_fd;
    char str[10];
    Hie_CID root_H_CID;
    int nid1, nid2;
    
    nid1 = rand() % 5000;
    nid2 = rand() % 5000;
    
    count = 0;                      //Nunber of nodes
    srand(time(NULL));              //Set the seed for rand()
    gaussian_rnd_init();            //Initialize Gaussian random number generator
    R = transmission_range(0);      //Determine communication range based on transmission power
    
    //Set all the node paraters. Some of the node location may not be used.
    //So they need to be set to 0
    for (i = 0; i < NODESX ; i++)   //All nodes in X direction
    {
        for(j = 0 ; j < NODESY ; j++)   //All nodes in Y direction
        {
            nodes[i][j].CID = 0;             // Initially set all nodes to 0
            nodes[i][j].NID = 0;             // Set NIDs from left to right
            nodes[i][j].no_broadcasts = 0;
            nodes[i][j].no_ACKs = 0;
            nodes[i][j].CH_NID = 0;
            nodes[i][j].tree_depth = 0;
            nodes[i][j].node_depth = 0;
            nodes[i][j].link_depth = 255;
            nodes[i][j].parent_CH_NID = 0;
            nodes[i][j].no_child_nodes = 0;
            nodes[i][j].no_routing_entries = 0;
            nodes[i][j].no_vsn_entries = 0;
            nodes[i][j].no_msg_forward = 0;
            nodes[i][j].node_dead = 0;
            nodes[i][j].energy = 0;
            nodes[i][j].marked_bcast_by_CID = 0;
            nodes[i][j].last_bcast_for_CID = 0;
            nodes[i][j].heard_ACK_for_CID = 0;
            nodes[i][j].send_ACK_for_CID = 0;
            nodes[i][j].send_tree_opt_msg = 255;
            nodes[i][j].no_CCHs = 0;
            nodes[i][j].in_event = 0;
            nodes[i][j].know_event = 0;
            nodes[i][j].send_routing_info = 0;
        }
    }

    //if nodes are to be generated randomly
    if(use_file == 0) 	
    {
        while (1)	//Loop forever
        {
            x = rand() % NODESX;    //Select random X
            y = rand() % NODESY;    //Select random Y
            if(nodes[x][y].NID == 0) //If node has not already being assigned
            {
                nodes[x][y].NID = y * NODESX + x + 1;   // Set NIDs from left to right. Start numbering from 1
                nodes[x][y].energy = E_NODE;		//Set the node's initial energy
                count++;                                //Increment no of nodes generated
            }
            if(count == NONODES)        //Break if require no of nodes are generated
                break;
        }
    }
    else                                //if previously generated node file to be used
    {
        nodes_fd = fopen(NODELIST, "r");    //Open file in read-only mode
        if(nodes_fd == NULL)                //If file is not open print error message
        {
            perror("ERROR:");
            exit(1);
        }

        for(i = 0; i < NONODES; i++)       //Read each node ID from file
        {
            fgets(str, 10, nodes_fd);
            NID = atoi(str);                //Convert to integer
        if(i == nid1)
               nid1 = NID;
if(i == nid2)
               nid2 = NID;
            
            x = (NID - 1) % NODESX;         //Given NID determine X & Y
            y = (NID - 1) / NODESX;
            nodes[x][y].NID = NID;          // Set NIDs from left to right on the grid. Set nos from 1
            nodes[x][y].energy = E_NODE;    //Set the nodes initial energy         
        }

        fclose(nodes_fd);                   //Close the file
    }

    if(nodes[STARTX][STARTY].NID == 0)      //See whether the starting node exist
    {
        printf("Can't continue. Intial node doesn't exist\n");
        exit(1);
    }

    //Add the first event to the event list
    start_NID = STARTY * NODESX + STARTX + 1;	//Determine the NID of the root node
    root_H_CID.id[0] = 0;			//Hierarchical cluster ID of the root node
    root_H_CID.id[1] = 0;
    root_H_CID.id[2] = 0;
    root_H_CID.id[3] = 0;

    //set the 1st cluster formation event. Event type 1.
    //Set start time as 1, CID as 1, no CH_NID since this node is the CH, hierarchical ID 
    //is 0 for 1st cluster, depth 0, no parent CH_NID or CID
    
    add_event(1, 1, start_NID, 1, start_NID, root_H_CID, 0, 0, MAX_TTL, 0, 0, root_H_CID);
    //add_event(1, 1, nid1, 1, nid1, root_H_CID, 0, 0, MAX_TTL, 0, 0, root_H_CID);
    //add_event(1, 2, nid2, 2, nid2, root_H_CID, 0, 0, MAX_TTL, 0, 0, root_H_CID);
    //next_cid++;
    //Add my own routing entry
    nodes[STARTX][STARTY].routing_table[0].valid = 1;   
    nodes[STARTX][STARTY].routing_table[0].H_CID = root_H_CID;
    nodes[STARTX][STARTY].routing_table[0].NID = start_NID;
    nodes[STARTX][STARTY].routing_table[0].learn_from = start_NID;
    nodes[STARTX][STARTY].routing_table[0].hops = 0;
    nodes[STARTX][STARTY].no_routing_entries++;     
}


/*------------------------------------------------------------------------------*
 * Following function add events to the Events List                             *
 * Implementation of this function will slightly vary depending on the tree     *
 * formation approach (breadth-first, depath first, etc.)                       *
 * This implementation is for the Breadth-first                                 *
 * event_type	- 	Intercluster or Intracluster event			*
 * start time 	-	starting time of the event. events are sorted		*
 * nid		- 	node ID of the event related node			*
 * cid 		- 	CID of the event related node				*
 * ch_nid	- 	CH NID. This will be 0 if the event is related to a new	*
 * 			cluser formation. Will be > 0 if event is within cluster*
 * h_cid	- 	Hierarchical cluster ID of the cluster			*
 * tree_depth	- 	depth of the node broadcasting the message based on 	*
 * 			cluster tree						*
 * node_depth	- 	depth of the node broadcasting the message		*
 * ttl		-	current TTL value of the cluster formation broadcast	*
 * parent_cid	-	CID of the parent CH					*
 * parent_ch_nid-	NID of the parent CH					*
 * parent_h_cid	- 	Hierarchical CID of the parent				*
 *------------------------------------------------------------------------------*/
void add_event(uchar event_type, int start_time, uint nid, uint cid, uint ch_nid, Hie_CID h_cid, 
        uchar tree_depth, uchar node_depth, uchar ttl, uint parent_cid, uint parent_ch_nid, Hie_CID parent_h_cid) 
{
    event *new, *current, *previous;

    if(root == NULL)			//If root of the event list is not defined
    {
        root = (event *)malloc(sizeof(event));
        if(root == NULL)		//If unable to allocate memory
        {
            perror("Error while allocating memory - in add_event function\n");
            exit(1);
        }
        root->next = NULL;
    }

    new = (event *)malloc(sizeof(event));	//New event
    if(new == NULL) 
    {
        perror("Error while allocating memory - in add_event function\n");
        exit(1);
    }

    //Set the parameters for the new event
    new->event_type = event_type;
    new->time = start_time;
    new->NID = nid;
    new->CID = cid;
    new->CH_NID = ch_nid;
    new->H_CID = h_cid;
    new->tree_depth = tree_depth;
    new->node_depth = node_depth;
    new->TTL = ttl;
    new->parent_CID = parent_cid;
    new->parent_CH_NID = parent_ch_nid;
    if(parent_ch_nid != 0)
        new->parent_H_CID = parent_h_cid;

    if(root->next == NULL)          //if event list is blank
    {
        root->next = new;
        new->next = NULL;
    }
    else                            //if event list contains at least one item
    {
        previous = root;
        current = root->next;

        while(1)                    //Find the proper location
        {
            if(current->time > start_time) 
            {
                new->next = current;    //Set the new event
                previous->next = new;
                break;
            }
            else if(current->next == NULL) 
            {
                current->next = new;
                new->next = NULL;
                break;
            }
            else
            {
                previous = current;
                current = current->next;
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Follwoing function remove the given event from the event list		*
 * start_time	-	remove the event related to the given start time	*
 * NID		- 	NID of the node related to the event			*
 *------------------------------------------------------------------------------*/
void remove_event(uchar event_type, uint start_time, uint nid) 
{
    event  *current, *previous;

    if(root->next == NULL)		//If event list is already empty
    {
        printf("Event list is already empty\n");
        exit(1);
    }

    //Locate the event with given time & NID
    previous = root;
    current = root->next;

    while (1) 
    {
        //If no more events is found break the loop
        if(current->next == NULL)
            break;

        //if exact event is found break the loop
        if((current->event_type == event_type) && (current->time == start_time) && (current->NID == nid))
            break;

        previous = current;
        current = current->next;
    }

    if(current->next != NULL)		//If event is the last event
    {
        previous->next = current->next;
        free(current);
    }
    else				//if event is the 1st event or in the middle
    {
        previous->next = NULL;
        free(current);
    }
}


/*------------------------------------------------------------------------------*
 * Process the event list starting from the first event, one event at a time	*
 * If no more events are available, print data about nodes & exit.		*
 *------------------------------------------------------------------------------*/
uchar process_event_list() 
{
    event *next_event;
    uint x, y, parentx, parenty, parent_nid, next_route_no, current_time, new_timeout;
    packet data_packet;
    Hie_CID tmp_h_cid;
    uchar result;
    
    tmp_h_cid.id[0] = 0;            //Intialize hirarchical ID
    tmp_h_cid.id[1] = 0;
    tmp_h_cid.id[2] = 0;
    tmp_h_cid.id[3] = 0;

    if(root->next == NULL)          //If no more events to handle print data & exit
        return 0;
    else 
    {       
        next_event = root->next;		//Get event at the top of the event list
        current_time = next_event->time;	//Set current time to the time of the event
        x = (next_event->NID - 1) % NODESX;	//X coordinates of node related to event
        y = (next_event->NID - 1) / NODESX;	//Y coordinates of node related to event

        //event related to a node broadcasting/forwarding a cluster formation broadcast.
        if(next_event->event_type == 1)		//Intracluster event
        {
            //Check for collisions
            if(next_event->next != NULL)	//If there are at least 2 events
            {
                //if two nodes are broadcasting at the same time & if collisions to be considered
                if((next_event->time == next_event->next->time) && (USE_COLLISIONS == 1)) 
                    mark_collision_region(next_event->NID, next_event->next->NID); //Mark the collision region
                else if(last_collision_set[1] != next_event->NID)   //If the node was not in a collision region
                    no_collision_nodes = 0;
            }

            if(MAX_TTL == next_event->TTL)	//If node is the CH. Has the highest TTL
            {
                if(nodes[x][y].CID == 0)     	//if not assigned to another cluser
                {
                    if(next_event->tree_depth != 254)	//if not a node trying to form a cluster by it self
                        add_nodes_to_cluster(current_time, next_event->NID, next_event->CID, next_event->NID, 
                                next_event->tree_depth, next_event->node_depth, next_event->TTL);
                    else
                        add_nodes_to_cluster(current_time, next_event->NID, next_event->CID, next_event->NID, 
                                next_event->tree_depth, next_event->node_depth,  1);

                    //Mark my last broadcast
                    nodes[x][y].last_bcast_for_CID = next_event->CID;
                    nodes[x][y].no_broadcasts++; //I send another broadcats
                    nodes[x][y].energy -= energy_to_transmit(CLUSTER_BCAST_SIZE, R); //Consume energy
                    if((optimized == 0) && (nodes[x][y].no_child_nodes > 0))	//If able to attract child node(s)
                    {
                        nodes[x][y].CID = next_event->CID;            //set the CID of the starting node
                        nodes[x][y].H_CID = next_event->H_CID;        //set the Hierarchical CID of starting node
                        nodes[x][y].CH_NID = next_event->NID;         //set me as my own CH
                        nodes[x][y].tree_depth = next_event->tree_depth;  //Set depth based on logical tree
                        nodes[x][y].node_depth = next_event->node_depth;  //set depth based on physical tree
                        nodes[x][y].parent_CH_NID = next_event->parent_CH_NID; //Keep track of parent

                         //Add routing enetries. Not valid if its root node
                        if(next_event->parent_CH_NID != 0)   //if not the root node
                        {   
                            //Add my own routing entry
                            nodes[x][y].routing_table[0].valid = 1;   
                            nodes[x][y].routing_table[0].H_CID = next_event->H_CID;
                            nodes[x][y].routing_table[0].NID = next_event->NID;
                            nodes[x][y].routing_table[0].learn_from = next_event->NID;
                            nodes[x][y].routing_table[0].hops = 0;
                            nodes[x][y].no_routing_entries++;     
                            //Add parent routing entry
                            nodes[x][y].routing_table[1].valid = 3;   
                            nodes[x][y].routing_table[1].H_CID = next_event->parent_H_CID;
                            nodes[x][y].routing_table[1].NID = next_event->parent_CH_NID;
                            nodes[x][y].routing_table[1].learn_from = next_event->parent_CH_NID;
                            nodes[x][y].routing_table[1].hops = 1;
                            nodes[x][y].no_routing_entries++;         
                            
                            //add my entry to parent routing table
                            parent_nid = nodes[x][y].parent_CH_NID;
                            parentx = (parent_nid - 1) % NODESX;
                            parenty = (parent_nid - 1) / NODESX;
                            next_route_no = nodes[parentx][parenty].no_routing_entries;
                            nodes[parentx][parenty].routing_table[next_route_no].valid = 1;
                            nodes[parentx][parenty].routing_table[next_route_no].H_CID = next_event->H_CID;
                            nodes[parentx][parenty].routing_table[next_route_no].NID = next_event->NID;
                            nodes[parentx][parenty].routing_table[next_route_no].learn_from = next_event->NID;
                            nodes[parentx][parenty].routing_table[next_route_no].hops = 1;
                            nodes[parentx][parenty].no_routing_entries++;
                        }

                        //Add a timeout event for this CH
                        new_timeout = current_time + TIMEOUT;
                        add_event(2, new_timeout, nodes[x][y].NID, nodes[x][y].CID, nodes[x][y].CH_NID, 
                                nodes[x][y].H_CID, nodes[x][y].tree_depth, nodes[x][y].node_depth, 0, 0, 0, tmp_h_cid);
                    }
                    else if(optimized == 1)			//Allow single node clusters in optimization phase
                    {
                        nodes[x][y].CID = next_event->CID;    //set the cid of the starting node
                        nodes[x][y].H_CID = next_event->H_CID;
                        nodes[x][y].CH_NID = next_event->NID; //set me as my own ch
                        nodes[x][y].tree_depth = next_event->tree_depth;
                        nodes[x][y].node_depth = next_event->node_depth;
                        nodes[x][y].parent_CH_NID = next_event->parent_CH_NID;
                    }
                }
                remove_event(1, current_time, next_event->NID);	//remove event
            }
            //If bcast message is within MAX_HOPS, add receiving nodes to cluster
            else if((MAX_TTL - next_event->TTL) < MAX_HOPS)     
            {
                if(nodes[x][y].CID == next_event->CID)    //if assigned to same cluser allow to add more nodes
                {
                    //Havent done the same broadcast earlier
                    if(nodes[x][y].last_bcast_for_CID != next_event->CID) 
                    {
                        add_nodes_to_cluster(current_time, next_event->NID, next_event->CID, 
                                next_event->CH_NID, next_event->tree_depth, next_event->node_depth, next_event->TTL);
                        nodes[x][y].energy -= energy_to_transmit(CLUSTER_BCAST_SIZE, R); 
                        nodes[x][y].last_bcast_for_CID = next_event->CID;
                        nodes[x][y].no_broadcasts++;
                    }
                }
                remove_event(1, current_time, next_event->NID);	//remove event
            }
            else if((MAX_TTL - next_event->TTL) < (MAX_TTL))	//if out side cluster
            {
                //if assigned to be bcasted by for the same cluster
                if(nodes[x][y].marked_bcast_by_CID == next_event->CID)    
                {
                    if(nodes[x][y].last_bcast_for_CID != next_event->CID)
                    {
                        forward_bcast_cluster(current_time, next_event->NID, next_event->CID, 
                                next_event->CH_NID, next_event->TTL, next_event->node_depth);
                        nodes[x][y].last_bcast_for_CID = next_event->CID;
                        nodes[x][y].no_broadcasts++;
                        nodes[x][y].energy -= energy_to_transmit(CLUSTER_BCAST_SIZE, R);
                    }
                }
                remove_event(1, current_time, next_event->NID);	//remove event
            }
            else	//I'm suppose to send an ACK as a CCH
            {
                //if assigned to be bcasted by for the same cluster & not already heard an ACK from a neighbor
                if(nodes[x][y].marked_bcast_by_CID == next_event->CID)
                    send_ACK_as_CCH(next_event->NID, next_event->CID, next_event->CH_NID);

                remove_event(1, current_time, next_event->NID);	//remove event
            }
        }
        else if (next_event->event_type == 2)	//event related to a CH timeout. Then select CCHs
        {
            if(nodes[x][y].no_CCHs > 255) 
                printf("Error: No of CCHs > 255. Overflow\n");

            select_child_CHs(current_time, nodes[x][y].no_CCHs, next_event->CID, next_event->CH_NID, 
                    next_event->H_CID, next_event->tree_depth, next_event->node_depth);
            remove_event(2, current_time, next_event->NID);	//remove event
        }
        else if (next_event->event_type == 3)	//Handle cluster optimization events
        {
            nodes[x][y].energy -= energy_to_transmit(CLUSTER_OPTI_SIZE, R);
            opti_cluster_tree(next_event->NID, next_event->CH_NID, next_event->tree_depth, 
                    next_event->node_depth, next_event->TTL);
            remove_event(3, current_time, next_event->NID);	//remove event
        }
        else if (next_event->event_type == 4)    //Event related to a VSN multicast
        {
            data_packet.source_NID = next_event->NID;
            data_packet.dest_NID = next_event->CH_NID;
            data_packet.source_H_CID.id[0] = next_event->H_CID.id[0];
            data_packet.source_H_CID.id[1] = next_event->H_CID.id[1];
            data_packet.source_H_CID.id[2] = next_event->H_CID.id[2];
            data_packet.source_H_CID.id[3] = next_event->H_CID.id[3];
            data_packet.dest_H_CID.id[0] = next_event->parent_H_CID.id[0];
            data_packet.dest_H_CID.id[1] = next_event->parent_H_CID.id[1];
            data_packet.dest_H_CID.id[2] = next_event->parent_H_CID.id[2];
            data_packet.dest_H_CID.id[3] = next_event->parent_H_CID.id[3];
            result = send_vsn_multicast_packet(data_packet);
            //Go back to caller & say failed. Caller assume 1 as sucess so don't give 1
            remove_event(4, current_time, next_event->NID);	//remove event
            if(result != 0)   
                return result + 1;
        }
        else if(next_event->event_type == 5)
        {
            add_ch_to_tree(next_event->NID, next_event->parent_CH_NID, next_event->tree_depth, 
                    next_event->node_depth, next_event->H_CID);
            remove_event(5, current_time, next_event->NID);	//remove event
        }
        else if(next_event->event_type == 6)
        {
            send_link_info(next_event->NID, next_event->tree_depth);
            remove_event(6, current_time, next_event->NID);	//remove event
        }
    }
    return 1;   //More event(s) in list
}


/*------------------------------------------------------------------------------*
 * Following function add nodes to a cluster    				*
 * Mark all the nodes in the communication range of the broadcasting node & 	*
 * without a cluster. In 1-hop cluster nid == ch_nid but this will be different	*
 * in multi-hop clusters.							*
 * start_time	- 	start time of the current event				*
 * nid		- 	NID of the node sending the bcast			*
 * cid		-	CID of the new clusters					*
 * ch_nid	-	NID of the CH						*
 * depth	- 	depth of the node sending the bcast			*
 * ttl		- 	TTL of the bcast					*
 *------------------------------------------------------------------------------*/
void add_nodes_to_cluster(uint start_time, uint nid, uint cid, uint ch_nid, uchar tree_depth, 
        uchar node_depth, uchar ttl) 
{
    int x, y, minx, miny, maxx, maxy, ch_x, ch_y, l, k;
    float distance;		//distance between reciveing & transmitting node
    char rec_rssi;              //Received signal strength
    uchar new_tree_depth, new_node_depth, new_ttl;
    uint new_start_time;
    Hie_CID tmp_h_cid;
    region my_region;

    x = (nid - 1) % NODESX;            //my X, Y coordinates
    y = (nid - 1) / NODESX;
    ch_x = (ch_nid - 1) % NODESX;	//X, Y coordinates of CH
    ch_y = (ch_nid - 1) / NODESX;

    my_region = get_node_region(x, y, R);	//determine my neighborhood
    minx = my_region.minx;
    miny = my_region.miny;
    maxx = my_region.maxx;
    maxy = my_region.maxy;

    tmp_h_cid.id[0] = 0;                //Temporary Hierarchical CID
    tmp_h_cid.id[1] = 0;
    tmp_h_cid.id[2] = 0;
    tmp_h_cid.id[3] = 0;

    //As the message travles depth increases & TTL reduces
    new_tree_depth = tree_depth + 1;	//Depth in logical tree
    new_node_depth = node_depth + 1;    //Depth in physical tree
    new_ttl = ttl - 1;                  //New TTL

    //set CID for only neighboring nodes
    for(l = miny; l <= maxy; l++) 
    {
        for (k = minx; k <= maxx; k++) 
        {
            //Cartesian distance. Then determine RSSI for given distance
            distance = sqrt((k - x)*(k-x)*(GRIDX * GRIDX) + (l - y)*(l-y)*(GRIDY * GRIDY));
            rec_rssi = RSSI(distance, 0);     
            
            // if within communication range, if the node exist & not the same node (distance != 0)
            if ((rec_rssi >= 0) && (nodes[k][l].NID != 0) && (distance != 0)) 
            {    
                //if not a member of a cluster or not in the collision region assigned to the current cluster
                if((nodes[k][l].CID == 0) && (is_in_collision(nodes[k][l].NID) == 0)) 
                {
                    nodes[k][l].CID = cid;
                    nodes[k][l].CH_NID = ch_nid;
                    nodes[k][l].tree_depth = new_tree_depth;
                    nodes[k][l].node_depth = new_node_depth;
                    nodes[ch_x][ch_y].no_child_nodes++;             //Add as a child node of CH
                    nodes[k][l].no_ACKs++;                          //I'm sending an ACK
                    nodes[k][l].energy -= energy_to_receive(CLUSTER_BCAST_SIZE);    //energy consumed to receive a message
                    nodes[k][l].energy -= energy_to_transmit(CLUSTER_ACK_SIZE, R);  //energy consumed to send ACK
                    nodes[ch_x][ch_y].energy -= energy_to_receive(CLUSTER_ACK_SIZE); //Energy to receive the ACK

                    //Waitsome random time before forwarding. Chose one of the Following lines if RSSI is used                                        
                    new_start_time = start_time + random_wait_time();
                    //new_start_time = start_time + rec_rssi * MAX_RND_TIME + random_wait_time();
                    if(new_start_time <= start_time)
                        new_start_time = start_time + 1;
                    
                    //Add to event list based on RSSI value. Add as type 1 event
                    if((new_ttl > 0) && (nodes[k][l].marked_bcast_by_CID != cid))	//if TTL has not expired
                    {
                        add_event(1, new_start_time, nodes[k][l].NID, cid, ch_nid, tmp_h_cid,
                            new_tree_depth, new_node_depth, new_ttl, 0, 0, tmp_h_cid);
                        nodes[k][l].marked_bcast_by_CID = cid;
                    }
                }
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Following function forward the cluster fomation broadcast                    *
 * start_time   -       start time of the current event                         *
 * nid          -       NID of the node sending the bcast                       *
 * cid          -       CID of the new clusters                                 *
 * ch_nid       -       NID of the CH                                           *
 * ttl          -       TTL of the bcast                                        *
 *------------------------------------------------------------------------------*/
void forward_bcast_cluster(uint start_time, uint nid, uint cid, uint ch_nid, uchar ttl, 
        uchar node_depth)
{
    int x, y, minx, miny, maxx, maxy, ch_x, ch_y, l, k;
    float distance;			//distance between reciveing & transmitting node
    uchar new_ttl;
    char rec_rssi;                      //Received RSSI
    uint new_start_time;
    Hie_CID tmp_h_cid;
    region my_region;

    ch_x = (ch_nid - 1) % NODESX;	//X, Y coordinates of CH
    ch_y = (ch_nid - 1) / NODESX;	
    x = (nid - 1) % NODESX;            //My X, Y coordinates
    y = (nid - 1) / NODESX;

    my_region = get_node_region(x, y, R);	//Get my region
    minx = my_region.minx;
    miny = my_region.miny;
    maxx = my_region.maxx;
    maxy = my_region.maxy;

    tmp_h_cid.id[0] = 0;
    tmp_h_cid.id[1] = 0;
    tmp_h_cid.id[2] = 0;
    tmp_h_cid.id[3] = 0;
    
    new_ttl = ttl - 1;			//As the message forwards, TTL reduces

    //set CID for only neighboring nodes
    for(l = miny; l <= maxy; l++) 
    {
        for (k = minx; k <= maxx; k++) 
        {
            //Distance & corrosponding RSSI value
            distance = sqrt((k - x)*(k-x)*(GRIDX * GRIDX) + (l - y)*(l-y)*(GRIDY * GRIDY));
            rec_rssi = RSSI(distance, 0);  //determine the RSSI for the signal for the given distance
            
            // if within communication range & node exist & not the same node (distance != 0)
            if ((rec_rssi >= 0) && (nodes[k][l].NID != 0) && (distance != 0))
            {
                //if not a member of a cluster & not used to send the same bcast use it to bcast the message
                if((nodes[k][l].CID == 0) && (nodes[k][l].marked_bcast_by_CID != cid) 
                        && (is_in_collision(nodes[k][l].NID) == 0)) 
                {  
                    nodes[k][l].energy -= energy_to_receive(CLUSTER_BCAST_SIZE);    //energy to receive a message       
                    
                    //Select one of the following lines depending on RSSI based forwarding or not
                    new_start_time = start_time + random_wait_time();                    
/*
                    if(new_ttl > 0)     //If not at the edge
                        new_start_time = start_time + rec_rssi * MAX_RND_TIME + random_wait_time();
                    else
                        new_start_time = start_time + (90 - rec_rssi) * MAX_RND_TIME + random_wait_time();
*/
                    if(new_start_time <= start_time)
                        new_start_time = start_time + 1;

                    add_event(1, new_start_time, nodes[k][l].NID, cid, ch_nid, tmp_h_cid, 0, 
                            (node_depth + 1), new_ttl, 0, 0, tmp_h_cid);
                    nodes[k][l].marked_bcast_by_CID = cid;
                }
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Send an acknoledgement to the CH indicating that node is a candiadte to be a *
 * a new CH. Consider all the nodes in the communication range of broadcasting	*
 * node & without a cluster. Nodes that has already heard a neighbor ACK will   *
 * not respond to the CH                                                        *
 * start_time   -       start time of the current event                         *
 * nid          -       NID of the node sending the bcast                       *
 * cid          -       CID of the new clusters                                 *
 * ch_nid       -       NID of the CH                                           *
 * ttl          -       TTL of the bcast                                        *
 *------------------------------------------------------------------------------*/
void send_ACK_as_CCH(uint nid, uint cid, uint ch_nid) 
{
    int x, y, minx, miny, maxx, maxy, ch_x, ch_y, l, k, i;
    float distance;		
    uint tmp_no_CCHs;
    region my_region;
    char rec_rssi;

    x = (nid - 1) % NODESX;            //X, Y coordinates of me
    y = (nid - 1) / NODESX;
    ch_x = (ch_nid - 1) % NODESX;	//X, Y coordinates of ch
    ch_y = (ch_nid - 1) / NODESX;	

    //Makesure that node has not already send a ACK for the same cluster or heard an ACK from a neighbor
    if((nodes[x][y].send_ACK_for_CID != cid) && (nodes[x][y].heard_ACK_for_CID != cid)) 
    {
        for(i = 0; i < nodes[ch_x][ch_y].no_CCHs; i++) //Makesure I have not already ACK
        {
            if(nid == nodes[ch_x][ch_y].CCHs[i])
                break;
        }
        if(i ==  nodes[ch_x][ch_y].no_CCHs)     //If no match add me as a CCH
        {
            tmp_no_CCHs = nodes[ch_x][ch_y].no_CCHs;
            nodes[ch_x][ch_y].CCHs[tmp_no_CCHs] = nid;
            if(nodes[ch_x][ch_y].no_CCHs > 200)
            {
                printf("No of candidate CHs overflow. Terminating.....\n");
                exit(1);
            }
            nodes[ch_x][ch_y].no_CCHs = tmp_no_CCHs + 1;
            nodes[x][y].send_ACK_for_CID = cid;
            nodes[x][y].heard_ACK_for_CID = cid;
            nodes[x][y].no_ACKs = nodes[x][y].no_ACKs + 3;  //ACK forwarded 3-hops
            //Energy to send & receive ACK
            nodes[x][y].energy -= energy_to_transmit(CLUSTER_ACK_SIZE, R);    
            nodes[ch_x][ch_y].energy -= energy_to_receive(CLUSTER_ACK_SIZE);
        }
        else
            return;             //if so discard
    }
    else			//if so discard
        return;

    my_region = get_node_region(x, y, R);	//get my region
    minx = my_region.minx;
    miny = my_region.miny;
    maxx = my_region.maxx;
    maxy = my_region.maxy;
    
    //Mark the neighbors indicating that they heard my ACK to the CH
    //set CID for only neighboring nodes
    for(l = miny; l <= maxy; l++) 
    {
        for (k = minx; k <= maxx; k++) 
        {
            distance = sqrt((k - x)*(k-x)*(GRIDX * GRIDX) + (l - y)*(l-y)*(GRIDY * GRIDY));
            rec_rssi = RSSI(distance, 0);
            
            // if within communication range & if the node exist & not the same node (distance != 0)
            if ((rec_rssi >= 0) && (nodes[k][l].NID != 0) && (distance != 0)) 
            {
                //if not a member of a cluster & marked to forward the bcast
                if((nodes[k][l].CID == 0) && (nodes[k][l].marked_bcast_by_CID == cid) 
                        && (is_in_collision(nodes[k][l].NID) == 0)) 
                {
                    nodes[k][l].heard_ACK_for_CID = cid;
                    nodes[k][l].energy -= energy_to_receive(CLUSTER_ACK_SIZE); //Cost of neighbors receiving the ACK
                }
            }
        }
    }
}


/*------------ -----------------------------------------------------------------*
 * Following function selects new CHs from list of available CCHs		*
 * no_cchs	- 	no of CCHs						*
 * depth	- 	depth of the parent CH					*
 * parent_cid	- 	CID of the parent CH					*
 * parent_h_cid	-	Hierarchical CID of the parent CH			*
 * parent_ch_nid-	NID of the parent CH					*
 *------------------------------------------------------------------------------*/
void select_child_CHs(uchar event_time, uchar no_cchs, uint parent_cid, uint parent_ch_nid, 
        Hie_CID parent_h_cid, uchar tree_depth, uchar node_depth) 
{
    uint no_new_chs, next_ch, j, l, new_time, x, y, cch_x, cch_y;
    uint selected_ch_list[10];
    uint no_selected_chs = 0;
    Hie_CID new_h_cid;

    if(no_cchs == 0)	//if no CCHs are there to be elected, just return back to caller
        return;

    x = (parent_ch_nid - 1) % NODESX;  //my X, Y coordinates
    y = (parent_ch_nid - 1) / NODESX;

    if(tree_depth == 0)                 //If depth 0 use maximim branching factor
        no_new_chs= NO_CCHS;
    else                                //else use only half of it
        no_new_chs= NO_CCHS/2;

    if(no_new_chs > no_cchs)		//if no of possible CCHs are less than what needs to be creted
        no_new_chs = no_cchs;		//generate the maximum possible no of CCHs

    for(j = 0; j < no_new_chs; j++)	//select given no of candidate neighbors as CHs
    {
        while(1) 
        {
            next_ch = rand() % no_cchs; //randomly select one of the nodes to be the CH
            for(l = 0 ; l < no_selected_chs ; l++)  //check whether it has been already selected
            {
                if(selected_ch_list[l] == nodes[x][y].CCHs[next_ch])	
                    break;
            }

            if(l != no_selected_chs)	//if already selected diacard & select another
                continue;

            //else select it as a new CH
            selected_ch_list[no_selected_chs] = nodes[x][y].CCHs[next_ch];
            no_selected_chs++;  //Increment no of new CHs
            next_cid++;		//generate the next cid
            new_h_cid = generate_CID(parent_h_cid, j, (tree_depth + 1));
            
            //set the timeing such that tree is formed using the breadth-first tree formation
            //This needs to be changed if depth-first tree formation is used
            if (j == 0)
                new_time = last_event_time((tree_depth + 1), NO_CCHS, TIMEOUT, DELAY_CCH) + 1 ;
            else
                new_time = last_event_time((tree_depth + 1), NO_CCHS, TIMEOUT, DELAY_CCH) + j * DELAY_CCH ;
            if(new_time <= event_time)          //Makesure new time is > old time
                new_time = event_time + 1;
            
            //add as a new CH event
            add_event(1, new_time, nodes[x][y].CCHs[next_ch], next_cid, 0, new_h_cid, 
                    (tree_depth + 1), (node_depth + MAX_TTL), MAX_TTL, parent_cid, parent_ch_nid, parent_h_cid);
            nodes[x][y].no_broadcasts = nodes[x][y].no_broadcasts + 3;  //Bcast is send 3-hops
            
            //Cost of sending the form_cluster function
            nodes[x][y].energy -= energy_to_transmit(CLUSTER_FORM_SIZE, R); 
            //Cost of receiving the form_cluster function
            cch_x = (nodes[x][y].CCHs[next_ch] - 1) % NODESX;
            cch_y = (nodes[x][y].CCHs[next_ch] - 1) / NODESX;
            nodes[cch_x][cch_y].energy -= energy_to_receive(CLUSTER_FORM_SIZE); 

            break;
        }
    }
}


/*------------------------------------------------------------------------------*
 * Following function optimize the cluster tree by broadcasting cluster 	*
 * optimization message								*
 * nid		- NID of the node						*
 * CH_nid	- NID of the CH sending the bcast				*
 * tree_depth	- depth of the CH broadcasting the message			*
 * node_depth	- depth of the node broadcasting or forwarding the bcast	*
 * ttl		- TTL of the message						*
 *------------------------------------------------------------------------------*/
void opti_cluster_tree(uint nid, uint CH_nid, uchar tree_depth, uchar node_depth, uchar ttl)
{
    int x, y, minx, miny, maxx, maxy, l, k;
    float distance, r;	
    Hie_CID tmp_H_CID;
    region my_region;
    char rec_rssi;

    x = (nid - 1) % NODESX;
    y = (nid - 1) / NODESX;
    
    if(USE_HP == 0) //If cluster optimization phase is high power
        r = transmission_range(0);
    else
        r = transmission_range(1);
    
    my_region = get_node_region(x, y, r);	//get my region
    minx = my_region.minx;
    miny = my_region.miny;
    maxx = my_region.maxx;
    maxy = my_region.maxy;

    tmp_H_CID.id[0] = 0;    //Hierarchical CID of the root node
    tmp_H_CID.id[1] = 0;
    tmp_H_CID.id[2] = 0;
    tmp_H_CID.id[3] = 0;

    if((ttl -1 ) < 0)       //if message expired
        return;

    for(l = miny; l <= maxy; l++)
    {
        for (k = minx; k <= maxx; k++) 
        {
            distance = sqrt((k - x)*(k-x)*(GRIDX * GRIDX) + (l - y)*(l-y)*(GRIDY * GRIDY));
            if(USE_HP == 0)
                rec_rssi = RSSI(distance, 0);
            else
                rec_rssi = RSSI(distance, 1);
                    
            //if in region, node exist and not the same node
            if((rec_rssi >= 0) && (nodes[k][l].NID != 0) && (distance != 0)) 
            {
                if(nodes[k][l].NID != nodes[k][l].CH_NID)   //if not a CH
                {
                    if(((ttl - 1) > 0) && (nodes[k][l].send_tree_opt_msg > (node_depth + 1))) 
                    {
                        last_type3_event_time++;
                        add_event(3, last_type3_event_time, nodes[k][l].NID, 0, CH_nid, tmp_H_CID, 
                                tree_depth, (node_depth + 1), (ttl - 1), 0, 0, tmp_H_CID);
                        nodes[k][l].no_broadcasts++;	//Broadcast cluster changes to child nodes
                        nodes[k][l].energy -= energy_to_receive(CLUSTER_OPTI_SIZE); //energy to receive message
                        nodes[k][l].send_tree_opt_msg = node_depth + 1;
                    }
                }
                else 
                {
                    //if a CH and current depth is higher
                    if(nodes[k][l].node_depth > (node_depth + 1)) 
                    {                        
                        nodes[k][l].tree_depth = tree_depth + 1;    //Set new depth & parent CH
                        nodes[k][l].node_depth = node_depth + 1;
                        nodes[k][l].parent_CH_NID = CH_nid;
                        nodes[k][l].no_ACKs = nodes[k][l].no_ACKs + (MAX_TTL - ttl);
                        nodes[k][l].energy -= energy_to_receive(CLUSTER_OPTI_SIZE);  //energy to receive message
                        nodes[x][y].energy -= energy_to_transmit(CLUSTER_ACK_SIZE, r);  //energy to send ACK
                    }
                    if(nodes[k][l].send_tree_opt_msg > (nodes[k][l].node_depth)) //If new message is better
                    {
                        last_type3_event_time++;
                        if(USE_HP == 0)
                            add_event(3, last_type3_event_time, nodes[k][l].NID, 0, nodes[k][l].NID, tmp_H_CID, 
                                nodes[k][l].tree_depth, nodes[k][l].node_depth, MAX_TTL, 0, 0, tmp_H_CID);
                        else
                            add_event(3, last_type3_event_time, nodes[k][l].NID, 0, nodes[k][l].NID, tmp_H_CID, 
                                nodes[k][l].tree_depth, nodes[k][l].node_depth, MAX_HOPS, 0, 0, tmp_H_CID);
                           
                        nodes[k][l].no_broadcasts++;	//Broadcast cluster changes to child nodes
                        nodes[k][l].send_tree_opt_msg = nodes[k][l].node_depth;
                    }
                }
            }
        }
    }
}


/*----------------------------------------------------------------------*
 * Allow nodes that are not in a cluster to join a neighboring cluster	*
 *----------------------------------------------------------------------*/
void opti_none_cluster_nodes()
{
    int i, j, k, l, minx, maxx, miny, maxy;
    region my_region;
    float distance;
    uchar my_exit = 0;
    char rec_rssi;

    for (j = 0; j < NODESY ; j++) //check for all the nodes
    {
        for(i = 0 ; i < NODESX ; i++) 
        {
            //if the node exist but not in a cluster
            if((nodes[i][j].NID != 0) && (nodes[i][j].CID == 0)) 
            {
                my_exit = 0;

                my_region = get_node_region(i, j, R);	//get my region
                minx = my_region.minx;
                miny = my_region.miny;
                maxx = my_region.maxx;
                maxy = my_region.maxy;

                for(l = miny; l <= maxy; l++) //in my region
                {
                    for (k = minx; k <= maxx; k++) 
                    {
                        distance = sqrt((k - i)*(k - i)*(GRIDX * GRIDX) + (l - j)*(l - j)*(GRIDY * GRIDY));
                        rec_rssi = RSSI(distance, 0);
                        
                        // if within communication range & if the node exist & a CH, join that cluster
                        if((rec_rssi >= 0) && (nodes[k][l].NID != 0) && (nodes[k][l].NID == nodes[k][l].CH_NID)) 
                        {
                            nodes[i][j].CID = nodes[k][l].CID;
                            nodes[i][j].CH_NID = nodes[k][l].CH_NID;
                            nodes[i][j].tree_depth = nodes[k][l].tree_depth + 1;
                            nodes[i][j].node_depth = nodes[k][l].node_depth + 1;
                            nodes[i][j].no_ACKs++;
                            nodes[k][l].no_child_nodes++;
                            nodes[i][j].energy -= energy_to_transmit(CLUSTER_ACK_SIZE, R);  //energy to send ACK
                            nodes[k][l].energy -= energy_to_receive(CLUSTER_ACK_SIZE);  //energy to receive ACK
                            my_exit = 1;			//Exit both loops
                            break;
                        }
                    }
                    if(my_exit == 1)	//exit 1st outer loop
                        break;
                }
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Following function update the child nodes if parent nodes changes its 	*
 * location in the cluster tree.						*
 *------------------------------------------------------------------------------*/
void update_child_nodes() 
{
    int k, l, CH_x, CH_y;
    uchar tree_depth_CH, node_depth_CH;

    for(l = 0; l < NODESY; l++) //check for all nodes
    {
        for (k = 0; k < NODESX; k++) 
        {
            //If I'm in a cluster and if I'm not a CH
            if((nodes[k][l].CID != 0) && (nodes[k][l].NID != nodes[k][l].CH_NID)) 
            {
                CH_x = (nodes[k][l].CH_NID - 1) % NODESX;
                CH_y = (nodes[k][l].CH_NID - 1) / NODESX;
                tree_depth_CH = nodes[CH_x][CH_y].tree_depth;
                node_depth_CH = nodes[CH_x][CH_y].node_depth;

                //My depth & my CH depth don't agree
                if(nodes[k][l].tree_depth != (tree_depth_CH + 1)) 
                {
                    nodes[k][l].tree_depth = tree_depth_CH  + 1;
                    nodes[k][l].node_depth = node_depth_CH  + 1;
                }
                if(nodes[k][l].node_depth != (node_depth_CH + 1)) 
                {
                    nodes[k][l].tree_depth = tree_depth_CH  + 1;
                    nodes[k][l].node_depth = node_depth_CH  + 1;
                }
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Form my own cluster if unable to join a neighboring cluster			*
 *------------------------------------------------------------------------------*/
void form_my_own_cluster() 
{
    int k, l, new_time;
    Hie_CID tmp_h_cid;

    tmp_h_cid.id[0] = 0;
    tmp_h_cid.id[1] = 0;
    tmp_h_cid.id[2] = 0;
    tmp_h_cid.id[3] = 0;

    for(l = 0; l < NODESY; l++)
    {
        for (k = 0; k < NODESX; k++) 
        {
            //if node exist & not in a cluster
            if((nodes[k][l].NID != 0) && (nodes[k][l].CID == 0)) 
            {
                new_time = rand() % RANDOM_WAIT;
                add_event(1, new_time, nodes[k][l].NID, next_cid, 0, tmp_h_cid, 254, 254, 
                        MAX_TTL, 0, 0, tmp_h_cid);
                next_cid++;
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Following functions randomly generate a socue and a destination node	& then	*
 * sends a message. It count the no of messages sucessfully delivered &         *
 * terminates when the 1st message get dropped. It indicate the reason why the  *
 * packet get dropped.                                                          *
 *------------------------------------------------------------------------------*/
void send_data()
{
    int s_x, s_y, d_x, d_y, source_CH_x, source_CH_y, dest_CH_x, dest_CH_y, CH_NID, i;
    uchar result;
    uint no_msg_delivered = 0;    //No of sucessfully delivered messages
    uint no_msg_dropped = 0;      //Nof of messages dropped
    uint no_route = 0;            //No fo routes not found
    packet data_packet;           //Data packet to be transmitted    
    double energy_before;
    
    energy_before = total_energy();
    //for(i = 0 ; i < NO_OF_PACKETS; i++)
    while (1)           //Loop until packet get dropped
    {
        while (1)	//Generate source node
        {
            s_x = rand() % NODESX;
            s_y = rand() % NODESY;
            //Make sure node is available & in a cluster
            if((nodes[s_x][s_y].NID == 0) || (nodes[s_x][s_y].CID == 0))	
                continue;
            else if(nodes[s_x][s_y].node_dead == 1) //Source node is dead find another
                continue;
            else //Form the source info of the data packet
            {
                data_packet.source_NID = nodes[s_x][s_y].NID;
                CH_NID = nodes[s_x][s_y].CH_NID;
                source_CH_x = (CH_NID - 1) % NODESX;	    
                source_CH_y = (CH_NID - 1) / NODESX;
                data_packet.source_H_CID = nodes[source_CH_x][source_CH_y].H_CID;
                break;
            }
        }	
        while (1)	//Generate destination node
        {
            d_x = rand() % NODESX;
            d_y = rand() % NODESY;
            //Make sure node is available & a member of a cluster
            if((nodes[d_x][d_y].NID == 0) || (nodes[d_x][d_y].CID == 0))
                continue;
            else if((s_x == d_x) && (s_y == d_y))   //if both source & destination is equal find another
                continue; 
            else     //Form the destination info of the data packet
            {
                data_packet.dest_NID = nodes[d_x][d_y].NID;
                CH_NID = nodes[d_x][d_y].CH_NID;
                dest_CH_x = (CH_NID - 1) % NODESX;	    
                dest_CH_y = (CH_NID - 1) / NODESX;
                data_packet.dest_H_CID = nodes[dest_CH_x][dest_CH_y].H_CID;
                break;
            }
        }
         
        result = send_data_packet(data_packet);    //Send data from random source to a random destination
        if(result == 1)          //If the message is dropped due to low energy
        {
            no_msg_dropped++;
            break;
        }
        else if (result == 2)    //If message drop due to the wrong route
        {
            no_route++;
            break;
        }
        else
            no_msg_delivered++;
     } 
    //printf("%d\n", no_msg_delivered);
    //printf("%d\t%d\t%d\n", no_msg_delivered, no_msg_dropped, no_route);
    printf("%d\t%f\t%f\t", no_msg_delivered, energy_before, total_energy());
}


/*------------------------------------------------------------------------------*
 * Following function try to deliver a message between a given source & a       *
 * destination. Function returns:                                               *
 * 0 - On sucess								*
 * 1 - when not enough energy to deliver the packet				*
 * 2 - When route to destination is not found 					*
 * data_packet - header of the data packet to be delivered                      *
 *------------------------------------------------------------------------------*/
unsigned char send_data_packet(packet data_packet)
{
    int s_x, s_y, d_x, d_y, source_CH_x, source_CH_y, dest_CH_x, dest_CH_y, receiver_x, 
            receiver_y, current_x, current_y;
    int neighbor_to_forward, current_NID; 
    int msg_send_by;	//Node that send the message

    receiver_x = -1;
    receiver_y = -1;
    msg_send_by = 0;

    s_x = (data_packet.source_NID - 1) % NODESX;	    
    s_y = (data_packet.source_NID - 1) / NODESX;
    d_x = (data_packet.dest_NID - 1) % NODESX;	    
    d_y = (data_packet.dest_NID - 1) / NODESX;
    source_CH_x = (nodes[s_x][s_y].CH_NID - 1) % NODESX;	    
    source_CH_y = (nodes[s_x][s_y].CH_NID - 1) / NODESX;
    dest_CH_x = (nodes[d_x][d_y].CH_NID - 1) % NODESX;	    
    dest_CH_y = (nodes[d_x][d_y].CH_NID - 1) / NODESX;
    
    //If souce is not a CH. Then forward the message to the CH
    if(data_packet.source_NID != nodes[s_x][s_y].CH_NID)	
    {
        nodes[s_x][s_y].energy -= energy_to_transmit(DATA_PACKET_SIZE , R);	
        if(nodes[s_x][s_y].energy < 0)	//If not enough energy
        {
            nodes[s_x][s_y].node_dead = 1; //Mark node as dead
            return 1;
        }
        //If the rceiving CH is dead, drop message
        if(nodes[source_CH_x][source_CH_y].node_dead == 1)	
            return 1;

        nodes[source_CH_x][source_CH_y].energy -= energy_to_receive(DATA_PACKET_SIZE);
        if(nodes[source_CH_x][source_CH_y].energy < 0) //If not enough energy to receive
        {
            nodes[source_CH_x][source_CH_y].node_dead = 1;	//Mark node as dead
            return 1;
        }
        msg_send_by = nodes[s_x][s_y].NID;
        nodes[s_x][s_y].no_msg_forward++;	//Another message is forwarded
    }

    //if source and destination has the same CH
    if(nodes[s_x][s_y].CH_NID == nodes[d_x][d_y].CH_NID)
    {
        nodes[source_CH_x][source_CH_y].energy -= energy_to_transmit(DATA_PACKET_SIZE , R);	
        if(nodes[source_CH_x][source_CH_y].energy < 0)	//If not enough energy
        {
            nodes[source_CH_x][source_CH_y].node_dead = 1; //Mark node as dead
            return 1;
        }
        nodes[source_CH_x][source_CH_y].no_msg_forward++;	//Another message

        if(nodes[d_x][d_y].node_dead == 1)	
            return 1;
        nodes[d_x][d_y].energy -= energy_to_receive(DATA_PACKET_SIZE);
        if(nodes[d_x][d_y].energy < 0)	//If not enough energy
        {
            nodes[d_x][d_y].node_dead = 1; //Mark node as dead
            return 1;
        }
        return 0;
    }

    current_NID = nodes[s_x][s_y].CH_NID;	// Start forwarding from the CH

    //Loop until destination CH is found
    while(1)
    {       
        //Find next hop
        neighbor_to_forward =  next_hop(data_packet.dest_H_CID, current_NID, msg_send_by);
        if (neighbor_to_forward == 0)     //No route to destination 
            return 2;	
        else if(neighbor_to_forward == current_NID)  //Same as destination
            break;
        else
        {	
            current_x = (current_NID - 1) % NODESX;	    
            current_y = (current_NID - 1) / NODESX;

            if(nodes[current_x][current_y].node_dead == 1)	//If is dead can't forward messages
                return 1;

            //Reduce energy to transmit. CH to CH messages are high power with within R*TTL_max
            nodes[current_x][current_y].energy -=  energy_to_transmit(DATA_PACKET_SIZE, (CH_CH_R_FACT * R));
            if(nodes[current_x][current_y].energy < 0)  //If no energy to tranmit
            {
                nodes[current_x][current_y].node_dead = 1;  //Mark node as dead
                return 1;                                   //Not enough energy
            }			     

            receiver_x = (neighbor_to_forward - 1) % NODESX;	    
            receiver_y = (neighbor_to_forward - 1) / NODESX;

            if(nodes[receiver_x][receiver_y].node_dead == 1) //Node is dead can't receive messages
                return 1;

            nodes[receiver_x][receiver_y].energy -= energy_to_receive(DATA_PACKET_SIZE);
            if(nodes[receiver_x][receiver_y].energy < 0)	//Not enough energy
            {
                nodes[receiver_x][receiver_y].node_dead = 1;  //Mark node as dead
                return 1;
            }
            msg_send_by = nodes[current_x][current_y].NID;
            nodes[current_x][current_y].no_msg_forward++;	//Another message is forwarded
            current_NID = neighbor_to_forward ; //Forward packet to neighbor. Neighbor becomes current node 
        }
    }
    if(current_NID != data_packet.dest_NID)
    {
        //current node x, y values are cacluated in the previous loop
        if(nodes[receiver_x][receiver_y].node_dead == 1)    //Node is dead can't forward message
            return 1;

        nodes[receiver_x][receiver_y].energy -= energy_to_transmit(DATA_PACKET_SIZE, R);	
        if(nodes[receiver_x][receiver_y].energy < 0)	//Not enough energy
        {	
            nodes[receiver_x][receiver_y].node_dead = 1;    //Mark node as dead
            return 1; 
        }		

        nodes[d_x][d_y].energy -= energy_to_receive(DATA_PACKET_SIZE);
        if(nodes[dest_CH_x][dest_CH_y].energy < 0)	//Not enough energy
        {
            nodes[d_x][d_y].node_dead = 1;          //Mark node as dead
            return 1;
        }
        nodes[receiver_x][receiver_y].no_msg_forward++;   //Another message is forwarded
    }
    return 0;	//Packet is sucessfully delivered
}


/*------------------------------------------------------------------------------*
 * Following function forms a VSN                                               *
 * Nodes in a given region form a VSN by send a message                         *                         
 *------------------------------------------------------------------------------*/
void form_vsn()
{
    int s_x, s_y, source_CH_x, source_CH_y, CH_NID, i, j;
    uchar ret_value;
    packet data_packet;
        
    for(i = 0 ; i < NO_EVENT_NODES; i++)
    {
        while (1)	
        {     
            //Enable one of the follwoing based on how VSN needs to be formed

            //Generate source node within entire region
            s_x = rand() % NODESX;
            s_y = rand() % NODESY;
            
            //Generate source node within 3 event regions        
/*
            if(!(((s_x >= 0) && (s_x <= 80) && (s_y >= 70) && (s_y <= 190)) ||
                    ((s_x >= 150) && (s_x <= 180) && (s_y >= 20) && (s_y <= 80)) ||
                    ((s_x >= 120) && (s_x <= 200) && (s_y >= 150) && (s_y <= 200))))
                continue;
*/
                     
/*
            //Generate source node within one event regions        
            if(!((s_x >= 0) && (s_x <= 80) && (s_y >= 70) && (s_y <= 190)))
                continue;
*/
            
            //Make sure node is available & in a cluster
            if((nodes[s_x][s_y].NID == 0) || (nodes[s_x][s_y].CID == 0))	
                continue;
            else if(nodes[s_x][s_y].node_dead == 1) //Source node is dead find another node
                continue;           
            else //Form the source info of the data packet
            {
                for(j = 0 ; j < i; j++) //skip if same event node is found
                {
                    if(event_nodes[j] == nodes[s_x][s_y].NID)
                        continue;
                }
                nodes[s_x][s_y].in_event = 1;   //In event region
                event_nodes[i] = nodes[s_x][s_y].NID; //Add to list of nodes in event

                data_packet.source_NID = nodes[s_x][s_y].NID;
                CH_NID = nodes[s_x][s_y].CH_NID;
                source_CH_x = (CH_NID - 1) % NODESX;	    
                source_CH_y = (CH_NID - 1) / NODESX;
                data_packet.source_H_CID = nodes[source_CH_x][source_CH_y].H_CID;
                
                data_packet.dest_NID = nodes[STARTX][STARTY].NID;   //Send to root node
                data_packet.dest_H_CID.id[0] = 0;
                data_packet.dest_H_CID.id[1] = 0;
                data_packet.dest_H_CID.id[2] = 0;
                data_packet.dest_H_CID.id[3] = 0;                
                break;
            }
        }
        
        ret_value= send_form_vsn_msg(data_packet);
        if(ret_value == 1)
            printf("Unable to send VSN formation message. Not enugh energy.\n");
        else if(ret_value == 2)
            printf("Unable to send VSN formation message. No route towards root node.\n");
        printf("%d\t", no_sin_hop);        
    }
    printf("%d\n",no_mulhop);
}


/*------------------------------------------------------------------------------*
 * Following function send a forms a VSN message                                *
 * Each node that detects an event send a message towards the roo node          *
 * If a CH has already send a message it will not send another                  *
 * If two events meet message will stop there & sending node will get info on   *
 * where they meet (Hierarchical address is send).                              *
 * data_packet - header of the VSN formation message                            *
 * return 0 on sucess, 1 if no energy & 2 is no route                           *
 *------------------------------------------------------------------------------*/
uchar send_form_vsn_msg(packet data_packet)
{
    int s_x, s_y, source_CH_x, source_CH_y, receiver_x, receiver_y, current_x, current_y;
    int neighbor_to_forward, current_NID, i; 
    int msg_send_by;	//Node that send the message
    uchar tmp_vsn_entries;
    
    receiver_x = -1;
    receiver_y = -1;
    msg_send_by = 0;
  
    s_x = (data_packet.source_NID - 1) % NODESX;	    
    s_y = (data_packet.source_NID - 1) / NODESX;
    if(nodes[s_x][s_y].know_event == 1)   //If already know event type 1
        return 0;
    else
        nodes[s_x][s_y].know_event = 1;
    
    source_CH_x = (nodes[s_x][s_y].CH_NID - 1) % NODESX;	    
    source_CH_y = (nodes[s_x][s_y].CH_NID - 1) / NODESX;
           
    //If souce is not a CH. Then forward the message to the CH
    if(data_packet.source_NID != nodes[s_x][s_y].CH_NID)	
    {
        nodes[s_x][s_y].energy -= energy_to_transmit(VSN_FORM_SIZE , R);	
        if(nodes[s_x][s_y].energy < 0)	//If not enough energy
        {
            nodes[s_x][s_y].node_dead = 1;  //Mark node as dead
            return 1;
        }
        nodes[s_x][s_y].know_event = 1;     //Know about event
        
        //If the rceiving CH is dead, drop message
        if(nodes[source_CH_x][source_CH_y].node_dead == 1)	
            return 1;

        nodes[source_CH_x][source_CH_y].energy -= energy_to_receive(VSN_FORM_SIZE);
        if(nodes[source_CH_x][source_CH_y].energy < 0) //If not enough energy to receive
        {
            nodes[source_CH_x][source_CH_y].node_dead = 1;	//Mark node as dead
            return 1;
        }
        
        //Add child node to VSN table
        tmp_vsn_entries = nodes[source_CH_x][source_CH_y].no_vsn_entries;
        //Makesure there are no duplicate entries
        for(i = 0 ; i < tmp_vsn_entries; i++)
        {
            //check for event type and NID
            if((nodes[source_CH_x][source_CH_y].vsn_table[i].NID == data_packet.source_NID) 
                    && (nodes[source_CH_x][source_CH_y].vsn_table[i].VSN == 1))
                break;
        }
        if(i == tmp_vsn_entries)
        {
            if(nodes[source_CH_x][source_CH_y].no_vsn_entries > MAX_VSN_ENTRIES)
            {
                printf("No of VSN entries overflow. Terminating.....\n");
                exit(1);
            }
            nodes[source_CH_x][source_CH_y].vsn_table[tmp_vsn_entries].NID = data_packet.source_NID;
            nodes[source_CH_x][source_CH_y].vsn_table[tmp_vsn_entries].VSN = 1;
            nodes[source_CH_x][source_CH_y].vsn_table[tmp_vsn_entries].node_type = 2; //Child node
            nodes[source_CH_x][source_CH_y].no_vsn_entries++;
        }
        
        nodes[s_x][s_y].no_msg_forward++;	//Another message is forwarded        
        
        no_sin_hop++;
        no_mulhop++;
        
        if(nodes[source_CH_x][source_CH_y].know_event == 1)//If CH already know event type 1
            return 0;
        else
            nodes[source_CH_x][source_CH_y].know_event = 1;        
        msg_send_by = nodes[s_x][s_y].NID;
    }

    current_NID = nodes[s_x][s_y].CH_NID;	// Start forwarding from the CH

    //Loop until destination CH is found
    while(1)
    {       
        //Find next hop
        neighbor_to_forward =  next_hop(data_packet.dest_H_CID, current_NID, msg_send_by);
        if(neighbor_to_forward == current_NID)  //Same as destination
            break;
        else if (neighbor_to_forward == 0)     //No route to destination 
            return 2;	
        else
        {	
            current_x = (current_NID - 1) % NODESX;	    
            current_y = (current_NID - 1) / NODESX;

            if(nodes[current_x][current_y].node_dead == 1)	//If is dead can't forward messages
                return 1;

            //Reduce energy to transmit. CH to CH messages are high power with within R*TTL_max
            nodes[current_x][current_y].energy -=  energy_to_transmit(VSN_FORM_SIZE, (CH_CH_R_FACT * R));
            if(nodes[current_x][current_y].energy < 0)  //If no energy to tranmit
            {
                nodes[current_x][current_y].node_dead = 1;  //Mark node as dead
                return 1;                                   //Not enough energy
            }
            
            //Add parent CH to VSN table
            tmp_vsn_entries = nodes[current_x][current_y].no_vsn_entries;
            //Makesure there are no duplicate entries
            for(i = 0 ; i < tmp_vsn_entries; i++)
            {
                //check for event type and NID
                if((nodes[current_x][current_y].vsn_table[i].NID == neighbor_to_forward) 
                        && (nodes[current_x][current_y].vsn_table[i].VSN == 1))
                    break;
            }
            if(i == tmp_vsn_entries)
            {
                nodes[current_x][current_y].vsn_table[tmp_vsn_entries].NID = neighbor_to_forward;
                nodes[current_x][current_y].vsn_table[tmp_vsn_entries].VSN = 1;
                nodes[current_x][current_y].vsn_table[tmp_vsn_entries].node_type = 1; //CH
                nodes[current_x][current_y].no_vsn_entries++;
            }

            receiver_x = (neighbor_to_forward - 1) % NODESX;	    
            receiver_y = (neighbor_to_forward - 1) / NODESX;

            if(nodes[receiver_x][receiver_y].node_dead == 1) //Node is dead can't receive messages
                return 1;

            nodes[receiver_x][receiver_y].energy -= energy_to_receive(VSN_FORM_SIZE);
            if(nodes[receiver_x][receiver_y].energy < 0)	//Not enough energy
            {
                nodes[receiver_x][receiver_y].node_dead = 1;  //Mark node as dead
                return 1;
            }
            
            //Add child CH to VSN table
            tmp_vsn_entries = nodes[receiver_x][receiver_y].no_vsn_entries;
            //Makesure there are no duplicate entries
            for(i = 0 ; i < tmp_vsn_entries; i++)
            {
                //check for event type and NID
                if((nodes[receiver_x][receiver_y].vsn_table[i].NID == current_NID) 
                        && (nodes[receiver_x][receiver_y].vsn_table[i].VSN == 1))
                    break;
            }
            if(i == tmp_vsn_entries)
            {
                nodes[receiver_x][receiver_y].vsn_table[tmp_vsn_entries].NID = current_NID;
                nodes[receiver_x][receiver_y].vsn_table[tmp_vsn_entries].VSN = 1;
                nodes[receiver_x][receiver_y].vsn_table[tmp_vsn_entries].node_type = 1; //CH
                nodes[receiver_x][receiver_y].no_vsn_entries++;
            }
            nodes[current_x][current_y].no_msg_forward++;	//Another message is forwarded
            if(nodes[receiver_x][receiver_y].know_event == 1)   //If already know event
                return 0;
            else
                nodes[receiver_x][receiver_y].know_event = 1;
            
            msg_send_by = nodes[current_x][current_y].NID;    
            current_NID = neighbor_to_forward ; //Forward packet to neighbor. Neighbor becomes current node
            no_sin_hop++;
            no_mulhop += 3;
        }
    }
    
    return 0;	//Packet is sucessfully delivered  
    
}


/*------------------------------------------------------------------------------*
 * Following functions sends VSN data packets from randomly selected node to    *
 * another randomly selected node.                                              *
 * It count the no of messages sucessfully delivered & terminates either when   *
 * the 1st message get dropped or given number of messages are transmitted      *
 *------------------------------------------------------------------------------*/
void send_vsn_unicast_data()
{
    int s_x, s_y, d_x, d_y, source_CH_x, source_CH_y, dest_CH_x, dest_CH_y, i;
    uint s_nid, d_nid, rnd, ch_nid;
    uchar result;
    uint no_msg_delivered = 0;    //No of sucessfully delivered messages
    uint no_msg_dropped = 0;      //Nof of messages dropped
    uint no_route = 0;            //No fo routes not found
    packet data_packet;           //Data packet to be transmitted       
    double energy_before, energy_after;
    
    energy_before = total_energy();
    for(i = 0 ; i < NO_OF_PACKETS; i++)  //Send NO_PACKETS
    {
        while(1)
        {
            //Generate source node
            rnd = rand() % NO_EVENT_NODES;
            s_nid = event_nodes[rnd];
            rnd = rand() % NO_EVENT_NODES;
            d_nid = event_nodes[rnd];
            
            if(s_nid == d_nid)  //If source & destination is same
                continue;
        
            s_x = (s_nid - 1) % NODESX;	    
            s_y = (s_nid - 1) / NODESX;
            ch_nid = nodes[s_x][s_y].CH_NID;
            source_CH_x = (ch_nid - 1) % NODESX;	    
            source_CH_y = (ch_nid - 1) / NODESX;            
                    
            d_x = (d_nid - 1) % NODESX;	    
            d_y = (d_nid - 1) / NODESX;
            ch_nid = nodes[d_x][d_y].CH_NID;        
            dest_CH_x = (ch_nid - 1) % NODESX;	    
            dest_CH_y = (ch_nid - 1) / NODESX;
        
            data_packet.source_NID = s_nid;
            data_packet.dest_NID = d_nid;
            
            data_packet.source_H_CID = nodes[source_CH_x][source_CH_y].H_CID;          
            data_packet.dest_H_CID = nodes[dest_CH_x][dest_CH_y].H_CID;
            break;
        }     

        //Send data from source to destination. Send as normal data packets
        result = send_data_packet(data_packet);    
        if(result == 1)          //If the message is dropped due to low energy
        {
            no_msg_dropped++;
            break;
        }
        else if(result == 2)    //If message drop due to the wrong route
        {
            no_route++;
            break;
        }
        else
            no_msg_delivered++;
     }
    energy_after = total_energy();
    //printf("%d\n", no_msg_delivered);
    //printf("%d\t%d\t%d\n", no_msg_delivered, no_msg_dropped, no_route); 
    printf("%d\t%f\t%f\n", no_msg_delivered, energy_before, energy_after); 
}


/*------------------------------------------------------------------------------*
 * Following function generates a VSN multicast packet and add it for routing   *
 * to the event list.                                                           *
 * After calling this function process_event() list must be caled to deliver    *
 * multicast messages. Delivery will be handled by the send_vsn_multicast_data()*
 *------------------------------------------------------------------------------*/
void send_vsn_multicast_data()
{
    int s_x, s_y, d_x, d_y, source_CH_x, source_CH_y, i, j, nid, ch_nid, des_nid, rnd;
    Hie_CID source_h_cid, dest_h_cid;
    uchar result;
    uint no_msg_delivered = 0;    //No of sucessfully delivered messages
    uint no_msg_dropped = 0;      //Nof of messages dropped
    uint no_route = 0;            //No fo routes not found
    double energy_before, energy_after;
    
    energy_before = total_energy();
    for(i = 0 ; i < NO_OF_PACKETS; i++)
    {     
        //Generate source node
        rnd = rand() % NO_EVENT_NODES;
        nid = event_nodes[rnd];       
        s_x = (nid - 1) % NODESX;	    
        s_y = (nid - 1) / NODESX;

        ch_nid = nodes[s_x][s_y].CH_NID;
        source_CH_x = (ch_nid - 1) % NODESX;	    
        source_CH_y = (ch_nid - 1) / NODESX;

        if(nodes[s_x][s_y].NID != nodes[s_x][s_y].CH_NID) //If not a CH
        {
            source_h_cid.id[0] = 0;
            source_h_cid.id[1] = 0;
            source_h_cid.id[2] = 0;
            source_h_cid.id[3] = 0;
            dest_h_cid = nodes[source_CH_x][source_CH_y].H_CID;

            //Parameters in the event list has following meanings
            //nid - Nid of source node
            //ch_nid - NID of destination node
            //h_cid - Hierarchical CID of source
            //parent_h_cid - - Hierarchical CID of destination
            add_event(4, last_type3_event_time, nid, 0, ch_nid, source_h_cid, 0, 0, 0, 0, 0, dest_h_cid);
            last_type3_event_time++;
        }
        else    //if CH send a seperate packet for each entry in VSN table
        {
            for(j = 0 ; j < nodes[s_x][s_y].no_vsn_entries; j++)
            {
                des_nid = nodes[s_x][s_y].vsn_table[j].NID;
                d_x = (des_nid - 1) % NODESX;	    
                d_y = (des_nid - 1) / NODESX;

                source_h_cid = nodes[source_CH_x][source_CH_y].H_CID;

                if(des_nid == nodes[d_x][d_y].CH_NID)   //If destination is a CH
                    dest_h_cid = nodes[d_x][d_y].H_CID;
                else    //if it's a child node
                {
                    dest_h_cid.id[0] = 0;
                    dest_h_cid.id[1] = 0;
                    dest_h_cid.id[2] = 0;
                    dest_h_cid.id[3] = 0;    
                }

                //Parameters in the event list has following meanings
                //nid - Nid of source node
                //ch_nid - NID of destination node
                //h_cid - Hierarchical CID of source
                //parent_h_cid - - Hierarchical CID of destination
                add_event(4, last_type3_event_time, nid, 0, des_nid, source_h_cid, 0, 0, 0, 0, 0, dest_h_cid);
                last_type3_event_time++;                                  
            }
        }
        while(1)    //Process the event list until no events are found
        {
            result = process_event_list(); 	                    
            if(result != 1) //Actual result is incremented by 1, by the sender
                break;
        }
        if(result == 2)          //If the message is dropped due to low energy
        {
            no_msg_dropped++;
            break;
        }
        else if (result == 3)    //If message drop due to the wrong route
        {
            no_route++;
            break;
        }
        else
            no_msg_delivered++;  
    }
    energy_after = total_energy();
    //printf("%d\n", no_msg_delivered);
    //printf("%d\t%d\t%d\n", no_msg_delivered, no_msg_dropped, no_route); 
    printf("%d\t%f\t%f\n", no_msg_delivered, energy_before, energy_after); 
}


/*------------------------------------------------------------------------------*
 * Following function forwards a VSN multicats message between a given source   *
 * & a destination. If the receiver has VSN entries in it's VSN table new events*
 * are added to the event list.                                                 *
 * data_packet - header of the VSN formation message                            *
 * return 0 on sucess, 1 if no energy & 2 is no route                           *
 *------------------------------------------------------------------------------*/
uchar send_vsn_multicast_packet(packet data_packet)
{
    int s_x, s_y, d_x, d_y, i, des_nid;
    Hie_CID source_h_cid, dest_h_cid;
       
    s_x = (data_packet.source_NID - 1) % NODESX;	    
    s_y = (data_packet.source_NID - 1) / NODESX;
    d_x = (data_packet.dest_NID - 1) % NODESX;	    
    d_y = (data_packet.dest_NID - 1) / NODESX;
    
    //receive the message
    if(nodes[s_x][s_y].NID != nodes[s_x][s_y].CH_NID)    //if source is a child node
    {
        if(nodes[s_x][s_y].node_dead == 1)	//If is dead can't forward messages
            return 1;

        //Reduce energy to transmit. CH to CH messages are high power with within R*TTL_max
        nodes[s_x][s_y].energy -=  energy_to_transmit(DATA_PACKET_SIZE, R);
        if(nodes[s_x][s_y].energy < 0)  //If no energy to tranmit
        {
            nodes[s_x][s_y].node_dead = 1;  //Mark node as dead
            return 1;                       //Not enough energy
        }    
        
        if(nodes[d_x][d_y].node_dead == 1) //Node is dead can't receive messages
            return 1;

        nodes[d_x][d_y].energy -= energy_to_receive(DATA_PACKET_SIZE);
        if(nodes[d_x][d_y].energy < 0)	//Not enough energy
        {
            nodes[d_x][d_y].node_dead = 1;  //Mark node as dead
            return 1;
        }
    }
    else    //if source is a another CH
    {
        if(nodes[d_x][d_y].NID != nodes[d_x][d_y].CH_NID)   //If destination is a child node
        {
            if(nodes[s_x][s_y].node_dead == 1)	//If is dead can't forward messages
                return 1;

            //Reduce energy to transmit. CH to CH messages are high power with within R*TTL_max
            nodes[s_x][s_y].energy -=  energy_to_transmit(DATA_PACKET_SIZE, R);
            if(nodes[s_x][s_y].energy < 0)  //If no energy to tranmit
            {
                nodes[s_x][s_y].node_dead = 1;  //Mark node as dead
                return 1;                       //Not enough energy
            }    

            if(nodes[d_x][d_y].node_dead == 1) //Node is dead can't receive messages
                return 1;

            nodes[d_x][d_y].energy -= energy_to_receive(DATA_PACKET_SIZE);
            if(nodes[d_x][d_y].energy < 0)	//Not enough energy
            {
                nodes[d_x][d_y].node_dead = 1;  //Mark node as dead
                return 1;
            }           
            return 0;   //Sucessfully delivered to destination
        }
        else    //if destination is a CH
        {
            if(nodes[s_x][s_y].node_dead == 1)	//If is dead can't forward messages
                return 1;

            //Reduce energy to transmit. CH to CH messages are high power with within R*TTL_max
            nodes[s_x][s_y].energy -=  energy_to_transmit(DATA_PACKET_SIZE, (CH_CH_R_FACT * R));
            if(nodes[s_x][s_y].energy < 0)  //If no energy to tranmit
            {
                nodes[s_x][s_y].node_dead = 1;  //Mark node as dead
                return 1;                       //Not enough energy
            }    

            if(nodes[d_x][d_y].node_dead == 1) //Node is dead can't receive messages
                return 1;

            nodes[d_x][d_y].energy -= energy_to_receive(DATA_PACKET_SIZE);
            if(nodes[d_x][d_y].energy < 0)	//Not enough energy
            {
                nodes[d_x][d_y].node_dead = 1;  //Mark node as dead
                return 1;
            }       
        }                
    }
    
    //Now source and destination changes. New destination is what is in my VSN table
    //Code will come this point only if it's a CH.
    s_x = d_x;  
    s_y = d_y;
    
    for(i = 0 ; i < nodes[s_x][s_y].no_vsn_entries; i++)
    {
        des_nid = nodes[s_x][s_y].vsn_table[i].NID;
        //Skip if the entry is for sender. Prevents loops
        if(des_nid == data_packet.source_NID)
            continue;
        
        d_x = (des_nid - 1) % NODESX;	    
        d_y = (des_nid - 1) / NODESX;
  
        source_h_cid = nodes[s_x][s_y].H_CID;

        if(des_nid == nodes[d_x][d_y].CH_NID)   //If destination is a CH
            dest_h_cid = nodes[d_x][d_y].H_CID;
        else    //if it's a child node
        {
            dest_h_cid.id[0] = 0;
            dest_h_cid.id[1] = 0;
            dest_h_cid.id[2] = 0;
            dest_h_cid.id[3] = 0;    
        }

        //Parameters in the event list has following meanings
        //nid - Nid of source node
        //ch_nid - NID of destination node
        //h_cid - Hierarchical CID of source
        //parent_h_cid - - Hierarchical CID of destination
        add_event(4, last_type3_event_time, nodes[s_x][s_y].NID, 0, des_nid, source_h_cid, 0, 0, 0, 0, 0, dest_h_cid);
        last_type3_event_time++;
    }
    return 0;
} 


/*------------------------------------------------------------------------------*
 * Following function is used to inform neighboring CHs about a particular CH's *
 * Hierarchical address. This help to build corss links along cluster tree      *
 *------------------------------------------------------------------------------*/
void inform_neighbors()
{
    int x, y, minx, miny, maxx, maxy, l, k, i;        
    region my_region;
    float length, range;
    
    range = CH_CH_R_FACT * R;
    
    for(x = 0 ; x < NODESX; x++)    //Check for CHs
    {
        for(y = 0; y < NODESY; y++)
        {
              //If node exist and is a CH
            if((nodes[x][y].NID == nodes[x][y].CH_NID) && (nodes[x][y].NID != 0)) 
            {
                nodes[x][y].energy -=  energy_to_transmit(DATA_PACKET_SIZE, (CH_CH_R_FACT * R));

                my_region = get_node_region(x, y, range); //determine my neighborhood
                minx = my_region.minx;
                miny = my_region.miny;
                maxx = my_region.maxx;
                maxy = my_region.maxy;
             
                for(l = miny; l <= maxy; l++)	
                {
                    for (k = minx; k <= maxx; k++)
                    {
                        length = sqrt((k - x)*(k - x)*(GRIDX * GRIDX) + (l - y)*(l - y)*(GRIDY * GRIDY));

                        // If within communication range, if node exists, if node is a CH
                        if ((length <= range) && (nodes[k][l].NID != 0) && (nodes[k][l].NID == 
                                nodes[k][l].CH_NID) && (length != 0))
                        {
                            //Check whther the routing entry already exit (parent/child will exist)
                            for(i = 0; i < nodes[k][l].no_routing_entries; i++)	
                            {
                                if(nodes[k][l].routing_table[i].NID == nodes[x][y].NID)
                                    break;
                            }
                            if(i != nodes[k][l].no_routing_entries) //Skip if CH is already in
                                continue;
                            else    //Else add to the routing table
                            {
                                if(nodes[k][l].no_routing_entries > MAX_ROUTES)
                                    printf("Number of routing entries overflow.\n");

                                nodes[k][l].routing_table[i].NID = nodes[x][y].NID;
                                nodes[k][l].routing_table[i].H_CID = nodes[x][y].H_CID;
                                nodes[k][l].routing_table[i].learn_from = nodes[x][y].NID;
                                nodes[k][l].routing_table[i].valid = 5;	//Entry learn from neighbor, 5 = 101
                                nodes[k][l].routing_table[i].hops = 1;
                                nodes[k][l].no_routing_entries++;
                            }
                        }
                    }
                }
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Following function forms a nother cluster tree based on already existing CHs *
 * Multiple such trees can be build by modifiying this function & other related *
 * functions                                                                    *
 * Such trees will form a connected graph in the network & can faciliate better *
 * node-to-node routing                                                         *
 * tree_depth - if tree to be formed by combining nodes at a particular depth in*
 * the orginal cluster tree. Consider if    depth <= CH depth <= depth + 1      *
 * Function needs to be modified if depth is not restricted.                    *
 *------------------------------------------------------------------------------*/
void form_second_cluster_tree(uchar tree_depth)
{
    int x, y, node_list[100], nid;
    uchar no_nodes = 0;
    Hie_CID h_cid;
    
    for(x = 0; x < NODESX; x++) //Find list of nodes in given depth
    {
        for(y = 0; y < NODESY; y++)
        {   //if its a CH & in given depth
            if((nodes[x][y].NID == nodes[x][y].CH_NID) && (nodes[x][y].NID != 0) 
                    && (nodes[x][y].tree_depth == tree_depth))
            {
                printf("%d\n", nodes[x][y].NID);
                if(no_nodes < 100)
                {
                    node_list[no_nodes] = nodes[x][y].NID;
                    no_nodes++;
                }
                else
                    continue;
            }
        }
    }
    
    nid = node_list[(rand() % no_nodes)];    //Pick a random node            
    x = (nid - 1) % NODESX;	    
    y = (nid - 1) / NODESX;
    h_cid.id[0] = 0;
    h_cid.id[1] = 0;
    h_cid.id[2] = 0;
    h_cid.id[3] = 0;
    nodes[x][y].Link_H_CID = h_cid;
    nodes[x][y].link_depth = 0;
    //Parameters in the event list has following meanings
    //This is a type 5 event
    //nid - Nid of source CH
    //h_cid - Hierarchical CID of source CH
    //tree_depth - tree depth in the cluster tree
    //node_depth - tree depth in the new tree
    //parent_nid - nid of the parent CH  
    add_event(5, last_type3_event_time, nid, 0, 0, h_cid, tree_depth, 0, 0, 0, 0, h_cid);
    last_type3_event_time++;
    while(process_event_list());
}


/*------------------------------------------------------------------------------*
 * Following function add already existing CHs to a new clustr tree             *
 * This function needs to rerun again & again until tree finish spanning        *
 * Will be initially called by the form_second_cluster_tree function.           *
 * For simplicity this process is sequential (no parallel events                *
 * nid - NID of the CH sending the message                                      *
 * parent_nid - NID of the parent CH                                            *
 * tree_depth - Depth in the original tree                                      *
 * link_depth - depth in the new tree/link                                      *
 * h_cid - Hierarchical CID in the original cluster tree                        *
 * -----------------------------------------------------------------------------*/
void add_ch_to_tree(uint nid, uint parent_nid, uchar tree_depth, uchar link_depth, Hie_CID h_cid)
{
    int x, y, minx, miny, maxx, maxy, l, k;
    region my_region;
    float length;
    uchar child_no, new_link_depth;
    Hie_CID tmp_h_cid;
    
    child_no = 0;
    new_link_depth = link_depth + 1;    
    tmp_h_cid.id[0] = 0;
    tmp_h_cid.id[1] = 0;
    tmp_h_cid.id[2] = 0;
    tmp_h_cid.id[3] = 0;        
        
    x = (nid - 1) % NODESX;	    
    y = (nid - 1) / NODESX;
    my_region = get_node_region(x, y, (CH_CH_R_FACT * R));	//Get my region
    minx = my_region.minx;
    miny = my_region.miny;
    maxx = my_region.maxx;
    maxy = my_region.maxy;
    
    nodes[x][y].energy -=  energy_to_transmit(CLUSTER_BCAST_SIZE, (CH_CH_R_FACT * R));
    if(nodes[x][y].energy < 0)  //If no energy to tranmit
    {
        nodes[x][y].node_dead = 1;  //Mark node as dead
        return ;                       //Not enough energy
    }                   
        
    for(l = miny; l <= maxy; l++) //Check within my region
    {
        for (k = minx; k <= maxx; k++) 
        {
            //Concept of RSSI is not applicable here as far as R is known
            length = sqrt((k - x)*(k-x)*(GRIDX * GRIDX) + (l - y)*(l-y)*(GRIDY * GRIDY));
            // if within communication range
            if((length <= (CH_CH_R_FACT * R)) && (nodes[k][l].NID != 0) && (nodes[k][l].NID == nodes[k][l].CH_NID))
            {
                if((nodes[k][l].tree_depth == tree_depth) || (nodes[k][l].tree_depth == (tree_depth + 1)))
                {
                    if (nodes[k][l].NID != parent_nid)  //skip parent
                    {
                        if(nodes[k][l].node_dead == 1) //Node is dead can't receive messages
                            continue;
                        nodes[k][l].energy -= energy_to_receive(CLUSTER_BCAST_SIZE);
                        if(nodes[k][l].energy < 0)	//Not enough energy
                        {
                            nodes[k][l].node_dead = 1;  //Mark node as dead
                            continue;
                        }

                        if(nodes[k][l].link_depth > new_link_depth)
                        {
                            if(child_no < 7)    //can't add more than 8 child nodes
                            {
                                nodes[k][l].Link_H_CID = generate_CID(h_cid, (int)child_no, new_link_depth);
                                nodes[k][l].link_depth = new_link_depth;
                                child_no++;
                                //Parameters in the event list has following meanings
                                //This is a type 5 event
                                //nid - Nid of source CH
                                //h_cid - Hierarchical CID of source CH
                                //tree_depth - tree depth in the cluster tree
                                //node_depth - tree depth in the new tree
                                //parent_nid - nid of the parent CH
                                add_event(5, last_type3_event_time, nodes[k][l].NID, 0, 0, 
                                        nodes[k][l].Link_H_CID, tree_depth, new_link_depth, 0, 0, nid, tmp_h_cid);

                                last_type3_event_time++;
                                //send the ACK & its received by the parent
                                nodes[k][l].energy -=  energy_to_transmit(CLUSTER_ACK_SIZE, (CH_CH_R_FACT * R));
                                nodes[x][y].energy -= energy_to_receive(CLUSTER_ACK_SIZE);                                
                            }
                            else
                                continue;                        
                        }
                        else
                            continue;
                    }
                    else
                        continue;
                }
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Following function discover CHs at a given depth & depth + 1 in the cluster  *
 * tree. It then initiates sharing of roting information between them.          *
 * tree_depth - Depth in the original tree                                      *
 * -----------------------------------------------------------------------------*/
void discover_neighbors_of_link(uchar tree_depth)
{
    int x, y;
    Hie_CID h_cid;
    
    h_cid.id[0] = 0;
    h_cid.id[1] = 0;
    h_cid.id[2] = 0;
    h_cid.id[3] = 0;
    
    for(x = 0; x < NODESX; x++) //Find list of nodes in given depth
    {
        for(y = 0; y < NODESY; y++)
        {   //if its a CH & in given depth
            if((nodes[x][y].NID == nodes[x][y].CH_NID) && (nodes[x][y].NID != 0) && 
                    ((nodes[x][y].tree_depth >= tree_depth) && nodes[x][y].tree_depth <= (tree_depth + 1)))
            {
                //Parameters in the event list has following meanings
                //This is a type 6 event
                //nid - Nid of CH sending routing table
                //tree_depth - tree depth in the cluster tree
                add_event(6, last_type3_event_time, nodes[x][y].NID, 0, 0, h_cid, tree_depth, 0, 0, 0, 0, h_cid);
                last_type3_event_time++;                   
                nodes[x][y].send_routing_info = 0;
            }
        }
    }
    while(process_event_list());    //process untill all events are completed
}


/*------------------------------------------------------------------------------*
 * Following function share routing table info amonge CHs at a given depth &    *
 * depth + 1 in the cluster tree. Info is shared only if they are neighbors,    *
 * within the given depth range and if the new info is better than going through*
 * the cluster tree (will add entry if same as distance through routing table)  *  
 * nid - node broadcasting its routing table                                    * 
 * tree_depth - Depth in the original tree                                      *
 * -----------------------------------------------------------------------------*/
void send_link_info(uint nid, uchar tree_depth)
{
    int x, y, minx, maxx, miny, maxy, i, j, k, l, data_size;
    region my_region;
    float length;
    uchar hops, depth;
    Hie_CID h_cid;
    
    h_cid.id[0] = 0;
    h_cid.id[1] = 0;
    h_cid.id[2] = 0;
    h_cid.id[3] = 0;
        
    x = (nid - 1) % NODESX;	    
    y = (nid - 1) / NODESX;
    data_size = sizeof(router_entry) * nodes[x][y].no_routing_entries;
    
    if(nodes[x][y].send_routing_info == 1)  //Discard if no update happened
        return;
    
    if(nodes[x][y].node_dead == 1) //Node is dead can't receive messages
        return;
    nodes[x][y].energy -=  energy_to_transmit(data_size, (CH_CH_R_FACT * R));         
    if(nodes[x][y].energy < 0)	//Not enough energy
    {
        nodes[x][y].node_dead = 1;  //Mark node as dead
        return;
    }
                        
    my_region = get_node_region(x, y, (CH_CH_R_FACT * R));	//Get my region
    minx = my_region.minx;
    miny = my_region.miny;
    maxx = my_region.maxx;
    maxy = my_region.maxy;
    
    for(l = miny; l <= maxy; l++) //Check within my region
    {
        for (k = minx; k <= maxx; k++) 
        {
            //Node should exist, should be a CH, should be within given depth
            if((nodes[k][l].NID != 0) && (nodes[k][l].NID == nodes[k][l].CH_NID) &&
                    ((nodes[k][l].tree_depth >= tree_depth) && (nodes[k][l].tree_depth <= (tree_depth + 1))))
            {               
                length = sqrt((k - x)*(k-x)*(GRIDX * GRIDX) + (l - y)*(l-y)*(GRIDY * GRIDY));
                // if within communication range
                if((length <= (CH_CH_R_FACT * R)) && (length != 0)) 
                {
                    //Energy to receive
                    if(nodes[k][l].node_dead == 1) //Node is dead can't receive messages
                        return;
                    nodes[k][l].energy -=  energy_to_receive(data_size);
                    if(nodes[k][l].energy < 0)	//Not enough energy
                    {
                        nodes[k][l].node_dead = 1;  //Mark node as dead
                        continue;
                    }
                    
                    for(i = 0 ; i < nodes[x][y].no_routing_entries; i++) //for each routing entry
                    {
                        depth = nodes[x][y].routing_table[i].H_CID.id[3] & 63;                        
                        //skip the entries beyound the given depth range
                        if((depth < tree_depth) || (depth > (tree_depth + 1))) 
                            continue;    
                        else if(nodes[x][y].routing_table[i].valid == 0) //continue if invalid
                            continue;                        
                        else //if valid
                        {
                            //compare with each entry in the receiver
                            for(j = 0; j < nodes[k][l].no_routing_entries; j++)
                            {
                                depth = nodes[k][l].routing_table[j].H_CID.id[3] & 63;                        
                                //skip the entries beyound the given depth range
                                if((depth < tree_depth) || (depth > (tree_depth + 1))) 
                                    continue;   
                                else if(nodes[k][l].routing_table[j].valid == 0) //skip if invalid
                                    continue;                                                            
                                else if(nodes[k][l].routing_table[j].valid == 3) //skip my parent entry
                                    continue;
                                else if(nodes[x][y].routing_table[i].NID == nodes[k][l].routing_table[j].NID)
                                {
                                    //if same entry see whether new info is better
                                    //if new imformation is not useful
                                    if((nodes[x][y].routing_table[i].hops + 1) >= nodes[k][l].routing_table[j].hops)
                                        break;  //break inner loop if useless, can't be two matching entries
                                    else    //if new info is useful
                                    {
                                        nodes[k][l].routing_table[j].valid = 5; //valid & from neighbor
                                        nodes[k][l].routing_table[j].NID = nodes[x][y].routing_table[i].NID;
                                        nodes[k][l].routing_table[j].H_CID = nodes[x][y].routing_table[i].H_CID;
                                        nodes[k][l].routing_table[j].learn_from = nid;
                                        nodes[k][l].routing_table[j].hops = nodes[x][y].routing_table[i].hops + 1;
                                        //Parameters in the event list has following meanings
                                        //This is a type 6 event
                                        //nid - Nid of CH sending routing table
                                        //tree_depth - tree depth in the cluster tree
                                        add_event(6, last_type3_event_time, nodes[k][l].NID, 0, 0, h_cid, tree_depth, 0, 0, 0, 0, h_cid);
                                        last_type3_event_time++;       
                                        nodes[k][l].send_routing_info = 0;
                                        
                                        break;
                                    }
                                }
                            }
                            
                            if(j == nodes[k][l].no_routing_entries) //No matching entry
                            {
                                if(nodes[k][l].no_routing_entries < MAX_ROUTES)
                                {
                                    //determine distance through cluster tree
                                    hops = hop_distance(nodes[k][l].H_CID, nodes[x][y].H_CID);
                                    //if cluster tree distance is >= to what I learn from neighbor
                                    if(hops >= (nodes[x][y].routing_table[i].hops + 1))
                                    {
                                        nodes[k][l].routing_table[j].valid = 5; //valid & from neighbor
                                        nodes[k][l].routing_table[j].NID = nodes[x][y].routing_table[i].NID;
                                        nodes[k][l].routing_table[j].H_CID = nodes[x][y].routing_table[i].H_CID;
                                        nodes[k][l].routing_table[j].learn_from = nid;
                                        nodes[k][l].routing_table[j].hops = nodes[x][y].routing_table[i].hops + 1;
                                        nodes[k][l].no_routing_entries++;
                                        //Parameters in the event list has following meanings
                                        //This is a type 6 event
                                        //nid - Nid of CH sending routing table
                                        //tree_depth - tree depth in the cluster tree
                                        add_event(6, last_type3_event_time, nodes[k][l].NID, 0, 0, h_cid, tree_depth, 0, 0, 0, 0, h_cid);
                                        last_type3_event_time++;       
                                        nodes[k][l].send_routing_info = 0;
                                    }
                                }
                                else
                                    printf("Number of routing entries overflow.\n");
                            }
                        }                        
                    }
                }            
            }
        }
    }    
    nodes[x][y].send_routing_info = 1;
}


/*------------------------------------------------------------------------------*
 * follwing function calculates the maximum achievable circularity (MAC) for    *
 * each cluster & sump circularity to the circularity.txt file                  *
 * this functions works only up to 3-hop clusters. Can be extened to other cases*
 * file - if 1 write to 1st file else write to 2nd file (optimization phase)    *
 *------------------------------------------------------------------------------*/
void calculate_circularity(uchar file) 
{
    int i, j, x, y, minx, miny, maxx, maxy, l, k, retvalue;
    float length, in_cluster, outof_cluster, circularity;
    int nb_list[3][500];
    int level0, level1, level2, n, m, p, q;
    region my_region;

    FILE *circlefd;
    if(file == 1) //which file
        circlefd = fopen(CIRCLEFILE1, "w");
    else
        circlefd = fopen(CIRCLEFILE2, "w");

    if(circlefd == NULL)
        perror("ERROR: No circularity data will be written.....");
    
    for (i = 0; i < NODESX ; i++) //Check for a CH from all nodes
    {
        for(j = 0 ; j < NODESY ; j++) 
        {
            //make sure that the node is a CH
            if((nodes[i][j].NID == nodes[i][j].CH_NID) && (nodes[i][j].NID != 0)) 
            {
                in_cluster = 0;		//no of nodes inside cluster
                outof_cluster = 0;	//no of nodes outside cluster
                level0 = level1 = level2 = 0;	//check no of hops for single & multi-hop clustering

                x = (nodes[i][j].NID - 1) % NODESX;    //Get my X, Y coordinates
                y = (nodes[i][j].NID - 1) / NODESX;
                my_region = get_node_region(x, y, R);	//Get my region
                minx = my_region.minx;
                miny = my_region.miny;
                maxx = my_region.maxx;
                maxy = my_region.maxy;

                for(l = miny; l <= maxy; l++) //Check within my region
                {
                    for (k = minx; k <= maxx; k++) 
                    {
                        //Concept of RSSI is not applicable here as far as R is known
                        length = sqrt((k - x)*(k - x)*(GRIDX * GRIDX) + (l - y)*(l - y)*(GRIDY * GRIDY));
                        if((length <=  R) && (nodes[k][l].NID != 0))     // if within communication range
                        {
                            if(nodes[k][l].CID == nodes[i][j].CID)      //If with the same CID, inside
                                in_cluster++;
                            else                                        //Else outside
                                outof_cluster++;
                            nb_list[0][level0] = nodes[k][l].NID;
                            level0++;
                        }
                    }
                }
                if( (MAX_HOPS > 1) && (level0 != 0))    //If multi-hop clusters
                {
                    for(n=0; n < level0 ; n++) 
                    {
                        x = (nb_list[0][n] - 1) % NODESX;
                        y = (nb_list[0][n] - 1) / NODESX;
                        my_region = get_node_region(x, y, R); //get my region
                        minx = my_region.minx;
                        miny = my_region.miny;
                        maxx = my_region.maxx;
                        maxy = my_region.maxy;

                        for(l = miny; l <= maxy; l++) //within my region
                        {
                            for (k = minx; k <= maxx; k++) 
                            {
                                length = sqrt((k - x)*(k - x)*(GRIDX * GRIDX) + (l-y)*(l-y)*(GRIDY * GRIDY));
                                if((length <=  R) && (nodes[k][l].NID != 0))
                                {
                                    for(m = 0; m < level0; m++) //neighbors at level 1
                                    {
                                        if(nb_list[0][m] == nodes[k][l].NID)
                                            break;
                                    }
                                    for( p = 0; p < level1; p++) //neighbors at level 2
                                    {
                                        if(nb_list[1][p] == nodes[k][l].NID)
                                            break;
                                    }
                                    if((m == level0) && (p == level1)) 
                                    {
                                        if(nodes[k][l].CID == nodes[i][j].CID) 
                                            in_cluster++;   //If with same CID in cluster
                                        else                //not in cluster
                                            outof_cluster++;   
                                        nb_list[1][level1] = nodes[k][l].NID;
                                        level1++;
                                    }
                                }
                            }
                        }
                    }
                }
                if( (MAX_HOPS > 2) && (level1 != 0)) //if 3 hops or more
                {
                    for(n=0; n < level1 ; n++) 
                    {
                        x = (nb_list[1][n] - 1) % NODESX;
                        y = (nb_list[1][n] - 1) / NODESX;
                        my_region = get_node_region(x, y, R);	//get my region
                        minx = my_region.minx;
                        miny = my_region.miny;
                        maxx = my_region.maxx;
                        maxy = my_region.maxy;

                        for(l = miny; l <= maxy; l++) 
                        {
                            for (k = minx; k <= maxx; k++) 
                            {
                                length = sqrt((k - x)*(k - x)*(GRIDX * GRIDX) + (l-y)*(l-y)*(GRIDY * GRIDY));
                                if((length <=  R) && (nodes[k][l].NID != 0)) 
                                {
                                    for(m = 0; m < level0; m++) //level 0
                                    {
                                        if(nb_list[0][m] == nodes[k][l].NID)
                                            break;
                                    }
                                    for( p = 0; p < level1; p++) //level 1
                                    {
                                        if(nb_list[1][p] == nodes[k][l].NID)
                                            break;
                                    }
                                    for(q= 0; q < level2; q++) //level 2
                                    {
                                        if(nb_list[2][q] == nodes[k][l].NID)
                                            break;
                                    }

                                    if((m == level0) && (p == level1) && (q == level2)) 
                                    {
                                        if(nodes[k][l].CID == nodes[i][j].CID)
                                            in_cluster++;
                                        else
                                            outof_cluster++;
                                        nb_list[2][level2] = nodes[k][l].NID;
                                        level2++;
                                    }
                                }
                            }
                        }
                    }
                }
                //circularity = total inside / total nodes in range
                //for multihop clusters a node is considerd to be in range only 
                //if there is a path from node to CH 
                circularity = (in_cluster/(in_cluster + outof_cluster)) * 100;
                retvalue = sprintf(msg, "%f\n", circularity);
                //retvalue = sprintf(msg, "%d\t%f\n", nodes[i][j].CID, circularity);
                fputs(msg, circlefd);
            }
        }
    }
    fclose(circlefd);	//close file
}


/*------------------------------------------------------------------------------*
 * Following function dump remaining energy of each node to a text file		*
 *------------------------------------------------------------------------------*/
void print_cluster_energy()
{
    int i, j, retvalue;
    FILE *energyfd;
    
    energyfd = fopen(ENERGYFILE, "w");  //open file
    if(energyfd == NULL)
        perror("ERROR: No energy data will be written.....");
        
    //Following code write node energy to the text file
    for (i = 0; i < NODESY ; i++) 
    {
        for(j = 0 ; j < NODESX ; j++) 
        {
            if(nodes[j][i].NID != 0) //if node exist
            {
                if(nodes[j][i].energy > 0.0)
                    retvalue = sprintf(msg, "%f\n", nodes[j][i].energy);
                else
                    retvalue = sprintf(msg, "%f\n", 0.00);
                fputs(msg, energyfd);
            }
        }
    }
    fclose(energyfd);   //close file
}


/*------------------------------------------------------------------------------*
 * This function either printer the node status on the terminal or print node   *
 * data to a text file named nodes.txt                                          *
 * symbols:                                                                     *
 * '.' - Indicate grid points with nodes                                        *
 * 'o' - Indicate nodes with a CH                                               *
 * '?' - Indicare nodes clusters that don't have a represnetable symbol         *
 * Cluster symbol followed by a . indicate CHs                                  *
 * pnt_console  - print data to console                                         *
 * file - which file to use. 1 - 1st file, 2 - 2nd file (optimized              *
 *------------------------------------------------------------------------------*/
void print_nodes(uchar pnt_console, uchar file) 
{
    int i, j, retvalue;
    FILE *nodefd;
    
    if(file == 1)   //open file
        nodefd = fopen(NODEFILE1, "w");
    else
        nodefd = fopen(NODEFILE2, "w");

    if(nodefd == NULL)
        perror("ERROR: No node data will be written.....");
    
    //Following code write node data to the text file
    for (i = 0; i < NODESY ; i++) 
    {
        for(j = 0 ; j < NODESX ; j++)
        {
            if(nodes[j][i].NID != 0)
            {
                //form the text string
/*
                retvalue = sprintf(msg, "%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\n", nodes[j][i].NID, nodes[j][i].CID, 
                        nodes[j][i].CH_NID, nodes[j][i].parent_CH_NID, nodes[j][i].tree_depth, 
                        nodes[j][i].no_child_nodes, nodes[j][i].no_broadcasts, nodes[j][i].no_ACKs, nodes[j][i].node_depth);
*/

                retvalue = sprintf(msg, "%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\n", nodes[j][i].NID, nodes[j][i].CID, 
                        nodes[j][i].CH_NID, nodes[j][i].parent_CH_NID, nodes[j][i].tree_depth, 
                        nodes[j][i].no_child_nodes, nodes[j][i].no_broadcasts, nodes[j][i].no_ACKs, nodes[j][i].node_depth, 
                        nodes[j][i].in_event, nodes[j][i].know_event);

                fputs(msg, nodefd); //write to file
            }
        }
    }
    fclose(nodefd); //close file

    if(pnt_console == 1)    //if needs to print to console
    {
        for (i = 0; i < NODESY ; i++) 
        {
            for(j = 0 ; j < NODESX ; j++) 
            {
                if(nodes[j][i].NID == 0)
                    printf(". ");   //No node in this location
                else {
                    printf("%c", CID_to_symbol_mapping(nodes[j][i].CID));

                    if(nodes[j][i].NID == nodes[j][i].CH_NID)
                        printf(".");
                    else
                        printf(" ");
                }
            }
            printf("\n");
        }
    }
}


/*------------------------------------------------------------------------------*
 * Following function add a node to the collision list if its is in the         *
 * communication range of two broadcasting nodes    the same time.     		*
 * nid1		- NID of the first node                                         *
 * nid2		- NID of the second node					*
 *------------------------------------------------------------------------------*/
void mark_collision_region(uint nid1, uint nid2)
{
    float ch_distance, distance1, distance2;
    int x1, y1, x2, y2, minx, maxx, miny, maxy, l, k, i;
    region my_region;

    if(last_collision_set[1] != nid1)	//Reset the list if last collision is not related
        no_collision_nodes = 0;

    last_collision_set[0] = nid1;   //Set the last nodes related to the collision
    last_collision_set[1] = nid2;

    x1 = (nid1 - 1) % NODESX;	//X, Y coordinates of node related to event
    y1 = (nid1 - 1) / NODESX;		
    x2 = (nid2 - 1) % NODESX;	//X, Y coordinates of node related to event
    y2 = (nid2 - 1) / NODESX;		

    ch_distance = sqrt((x1 - x2)*(x1 - x2)*(GRIDX * GRIDX) + (y1 - y2)*(y1 - y2)*(GRIDY)*(GRIDY));
    if(ch_distance <= (2 * R))		//if within each others range
    {
        my_region = get_node_region(x1, y1, R);	//get my region
        minx = my_region.minx;
        miny = my_region.miny;
        maxx = my_region.maxx;
        maxy = my_region.maxy;

        for(l = miny; l <= maxy; l++) 
        {
            for (k = minx; k <= maxx; k++) 
            {
                distance1 = sqrt((k - x1)*(k - x1)*(GRIDX * GRIDX) + (l - y1)*(l - y1)*(GRIDY * GRIDY));
                if((distance1 <= R) && (nodes[k][l].NID != 0)) 
                {
                    distance2 = sqrt((k - x2)*(k - x2)*(GRIDX * GRIDX) + (l - y2)*(l - y2)*(GRIDY * GRIDY));
                    if((distance2 <= R) && (nodes[k][l].NID != 0)) 
                    {
                        //Check for overflow. If needed to change set the value in header file
                        if(no_collision_nodes < NO_COLLISION_NODES) 
                        {
                            for(i = 0; i < no_collision_nodes; i++) //Don't put the same node again & again
                            {
                                if(collision_nodes[i] == nodes[k][l].NID)
                                    break;
                            }
                            if(i == no_collision_nodes)	//if no match found add
                            {
                                //Add node to the collision region
                                collision_nodes[no_collision_nodes] = nodes[k][l].NID;	
                                no_collision_nodes++;	//Increment no of nodes in collision region
                            }
                        }
                        else 
                        {
                            printf("No of collision nodes overflow...\n");
                            exit(0);
                        }
                    }
                }
            }
        }
    }
}


/*------------------------------------------------------------------------------*
 * Following function checks whether a given node is in the collision range	*
 * nid		- NID of the node						*
 * Return 	- 1 if in the collision region & 0 if not			*
 *------------------------------------------------------------------------------*/
char is_in_collision(uint nid) 
{
    int i;

    for(i = 0; i < no_collision_nodes; i++) //Check to see whether node is in collsion list
    {
        if(collision_nodes[i] == nid)       //if so break
            break;
    }
    if(i != no_collision_nodes)
        return 1;			//if in collision region
    else
        return 0;
}


/*------------------------------------------------------------------------------*
 * Follwoing function returns the NID of the next hop to forward the message	*
 * If it is for the current cluster same NID is returned			*
 * If its for a neighboring cluster its NID is returned				*
 * Otherwice the NID of the next hop is returned based on the entries in the 	*
 * routing table.                                                               *
 * If no suitable next hop can't be found 0 is returned				*
 * dest_add  - Destination hierarchical address                                 *
 * current_NID - NID of the node trying to determine next hop                   *
 * sendder_NID - NID of the node that forwarded the message                     *
 *------------------------------------------------------------------------------*/
int next_hop(Hie_CID dest_add, int current_nid, int sender_nid)
{
    int x, y, i, j;
    nei_status neighbors[MAX_ROUTES];	//Hold hop count for each router table entry
    int no_routing_entries;
    uchar result;

    x = (current_nid - 1) % NODESX;
    y = (current_nid - 1) / NODESX;
     
    no_routing_entries = nodes[x][y].no_routing_entries;
    for(i = 0; i < no_routing_entries; i++) //For each entry in the routing table
    {            
        if((nodes[x][y].routing_table[i].valid & 1) == 0) //If route is invalid discard
            neighbors[i].hops = 255;	//Set as unreachable	     
        //Check whether the destination is my address or neighbor address
        else if((nodes[x][y].routing_table[i].H_CID.id[0] == dest_add.id[0]) //neighbor is destination
            && (nodes[x][y].routing_table[i].H_CID.id[1] == dest_add.id[1])
            && (nodes[x][y].routing_table[i].H_CID.id[2] == dest_add.id[2])
            && (nodes[x][y].routing_table[i].H_CID.id[3] == dest_add.id[3]))           
            return nodes[x][y].routing_table[i].learn_from;
        else if(nodes[x][y].routing_table[i].hops == 0) //Skip my entry
            neighbors[i].hops = 255;	//Set as unreachable	 
        else
        {
            result = hop_distance(nodes[x][y].routing_table[i].H_CID, dest_add);
            if(result == 0)
                neighbors[i].hops = 255;	//Set as unreachable	
            else
            {                          
                neighbors[i].hops = result + nodes[x][y].routing_table[i].hops;	
                neighbors[i].nei_NID = nodes[x][y].routing_table[i].learn_from;               
            }
        }
    }

    bubble_sort(neighbors, no_routing_entries);	// Sort based on hop count
    
    if(neighbors[0].hops == 255)    //No matching next hop found
        return 0;		
    else	
    {
        //If 2 or more best entries, skip parent entry
        if((neighbors[0].hops == neighbors[1].hops) && (neighbors[0].nei_NID != neighbors[1].nei_NID) 
                && (neighbors[0].nei_NID == nodes[x][y].parent_CH_NID) && (neighbors[1].nei_NID != sender_nid)) 
        {
            return neighbors[1].nei_NID;
        }
        else if (neighbors[0].nei_NID != sender_nid)	//If next hop is not same as the sender
            return neighbors[0].nei_NID;
        else    //If so pick the next best node
        {	
            for(j = 1 ; j < no_routing_entries; j++)  //Find a node which is not energy constrained
            {
                if((neighbors[j].hops < 255) && (neighbors[j].nei_NID != sender_nid)
                        && (neighbors[j].nei_NID != current_nid)) //If found stop
                    return neighbors[j].nei_NID;
            }
            if(j == no_routing_entries)		//No possible next node
		return 0;
        }
    }
    return 0;
}


/*------------------------------------------------------------------------------*
 * Following function add us the total energy remaining in the network          *
 * return total energy                                                          *
 *------------------------------------------------------------------------------*/
double total_energy()
{
    int i, j;
    double energy = 0.00;
    
    for(i = 0 ; i < NODESX; i++)
    {
        for(j = 0 ; j < NODESY; j++)
        {
            if(nodes[i][j].NID != 0)
                energy += (double)nodes[i][j].energy;
        }
    }
    return energy;
}

void who_died(uchar d)
{
    int x, y;
    
    for(x = 0 ; x < NODESX; x++)
    {
        for(y = 0 ; y < NODESY; y++)
        {
            if(nodes[x][y].node_dead == 1)
            {
                if((nodes[x][y].tree_depth == d) || (nodes[x][y].tree_depth == (d + 1)))
                    printf("0\n");
                else
                    printf("1\n");                
            }
        }
    }
}
