/*------------------------------------------------------------------------------*
 * Version - 6.0.1                                                              *
 * Last modified - 12/01/2007							*
 * BY - Dilum Bandara                                                           *
 * Type - HHC - Breadth First tree formation & hierarchical naming              *
 * Also support cluster formation based on predistributed keys			*
 *------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include "types.h"
#include "simulator.h"
#include "energy.h"
#include "print_data.h"
#include "kps.h"

node nodes[NODESX][NODESY];			//Hold the node information	
event *root;					//Root of the event list
uint candidateCHs[625];         			//Hold the candidate CHs
uint no_candidate_CHs = 0;       		//No of candidate CHs for each parent CH
char msg[100];					//Hold the temporaly string that is appened to a text file
uint nextCID = 1;				//CID of the 1st cluster. subsequently hold future CIDs
uint current_time  = 0;				//Hold the time of the event currently being processed
uint last_event_time = 0;			//Time of the last event	
packet data_packet;				
uint no_of_nodes = 0;				//No of sensor nodes
uint kps_p, kps_k, kps_z;			//k, p, z of key predistribution scheme

int main()
{
	init(USE_NODE_FILE);			//initialize nodes & link list
	while(1)				//Process the event list until no events are found
	{
		process_event_list();
	}
}

/*------------------------------------------------------------------------------*
 * Initialise the nodes & event list						*
 * Nodes can be either randomly genrated or assigned based on a pregenerated	*
 * file.									*
 * use_file - if 1 use the pregenerated node id file, else place nodes randomly *
 *------------------------------------------------------------------------------*/
void init(uchar use_file)
{
	uint count, x, y, i, j, start_nid, nid, z;
	FILE *nodes_fd;
	char str[10];
	Hie_CID root_H_CID;
	int retvalue;
	key_index key_node[MAX_Z]; 
	
	count = 0;
	srand(time(NULL));                              //Set the seed for rand()
	
	//Set all the nodes to 0
	for (i = 0; i < NODESX ; i++)
	{
		for(j = 0 ; j < NODESY ; j++)
		{
			nodes[i][j].CID = 0;                    // Initially set all nodes to 0
			nodes[i][j].NID = 0;                    // Set NIDs from left to right. Set nos from 1
			nodes[i][j].broadcasts = 0;
			nodes[i][j].CH_NID = 0;
			nodes[i][j].parent_CH_NID = 0;
			nodes[i][j].no_child_nodes = 0;
			nodes[i][j].no_routing_entries = 0;
			nodes[i][j].no_msg_forward = 0;
			nodes[i][j].sent_inform_neighbor = 0;
			nodes[i][j].node_dead = 0;
			nodes[i][j].no_key_blocks = 0;
			nodes[i][j].no_key_to_CH = 0;
		}
	}
	
	//Set the given no of nodes (Total grid points) / Probability of placing a node on a grid point 	
	no_of_nodes = (NODESX * NODESY) /  NODEPROB;
	//no_of_nodes = 7500;
	//no_of_nodes = 5550;

	if(use_file == 0)
	{
		while (1)
		{
			x = rand() % NODESX;		//Randomly pick a x coordinate
			y = rand() % NODESY;		//randomly pick a y coordinate
			if(nodes[x][y].NID == 0)	//If the node has already not being placed	
			{		
				nodes[x][y].NID = y * NODESX + x + 1;   // Set NIDs from left to right. Set nos from 1
				nodes[x][y].energy = E_NODE;		//Set the node's initial energy
				count++;			//Increment the no of nodes generated
			}
			if(count == no_of_nodes)		//Stop of required no of nodes are placed
				break;
		}

	        if(nodes[STARTX][STARTY].NID == 0)		//See whether the starting node exist
		{
			printf("Can't continue. Initial node doesn't exist\n");
			exit(1);
		}
	}
	else
	{
		nodes_fd = fopen(NODELIST, "r");		//Open the pregenrated node file
		if(nodes_fd == NULL)				//Unable to open print error message
		{
			perror("ERROR:");
			exit(1);
		}
	
		for(i = 0; i < 	no_of_nodes; i++)		//Extract each node from file & set appropriate node info
		{
			fgets(str, 10, nodes_fd);
			nid = atoi(str);
			x = (nid - 1) % NODESX;
			y = (nid - 1) / NODESX;
			nodes[x][y].NID = y * NODESX + x + 1;	// Set NIDs from left to right. Set nos from 1
			nodes[x][y].energy = E_NODE;		//Set the nodes initial energy
		}
		fclose(nodes_fd);				//Close the node file
	}

	//Add the first event to the event list
	start_nid = STARTY * NODESX + STARTX + 1;
	root_H_CID.id[0] = 0;
	root_H_CID.id[1] = 0;
	add_to_event_list(1, start_nid, 1, root_H_CID, 0, MAX_TTL, 0, 0, root_H_CID);

	if(USE_KPS == 1)					//If key predistribution to be used	
		assign_key_id();				//then assign keys nodes
	if((USE_KPS == 1) && (NODE_COMP == 1))			//If key predistribution to be used & nodes are compromized
	{
		count = 0;
		while(1)					//Loop untill all the nodes are added
		{ 
			x = rand() % NODESX;		//Randomly pick a x coordinate
			y = rand() % NODESY;		//randomly pick a y coordinate

			if(nodes[x][y].NID == 0)	//If the node has already not being placed	
				continue;
			else
			{
				for(z = 0; z < kps_z ; z++)
					key_node[z] = nodes[x][y].key_IDs[z];

				retvalue = add_compromized_blocks(key_node, kps_p, kps_k, kps_z);
				if(retvalue == 1)
					count++;
				else
				{
					printf("Unable to add more compromized nodes\n. Terminating.....");
					exit(1);
				}
			}
			if(count == COMP_NODES)
			break;
		}
	}
}
/*------------------------------------------------------------------------------*
 * Add events to event list							*
 * Implementation of this function will slightly vary depending on the tree	*
 * formation approach (breadth-first, depath first, etc.)			*
 * This implementation is for the Breadth-first					*
 * start time   -	starting time of the event. events are sorted		*
 * nid		-	node ID of the event related node 			*
 * cid		-	CID of the event related node				*
 * h_cid	-	Hierarchical cluster ID of the cluster			*
 * depth	-	depth of the node broadcasting the message		* 
 * ttl		-	current TTL value of the cluster formation broadcast	*
 * parent_cid	-	CID of the parent CH					*
 * parent_CH_NID-	NID of the parent CH					*
 * parent_H_CID -	Hierarchical CID of the parent				*
 *------------------------------------------------------------------------------*/
void add_to_event_list(uint start_time, uint nid, uint cid, Hie_CID h_cid, uchar depth, uchar ttl, uint parent_cid, uint parent_ch_nid, Hie_CID parent_h_cid)
{
	event *new, *current, *previous;

	if(root == NULL)					//If root of the event list is not already set
	{
		root = (event *)malloc(sizeof(event));		//Unable to allocate memory
		if(root == NULL)
		{
			perror("Error while allocating memory\n");
			exit(1);
		}
		root->next = NULL;				//Set the next event to NULL
	}

	last_event_time = start_time;				//Always add as the last event
	current = root;
	previous = NULL;

	while( (current->next != NULL) && (current->time < start_time))	//Locate the position to place the event within the list
	{
		previous = current;
		current = current->next;
	}			
	new = (event *)malloc(sizeof(event));			//Allocate memory for the new event
	if(new == NULL)
	{
		perror("Error while allocating memory\n");									
		exit(1);
	}
	new->time = start_time;					//Set parameters of the new event
	new->NID = nid;
	new->CID = cid;
	new->H_CID = h_cid;
	new->depth = depth;
	new->ttl = ttl;
	new->parent_CID = parent_cid;
	new->parent_CH_NID = parent_ch_nid;
	if(parent_ch_nid != 0)
		new->parent_H_CID = parent_h_cid;
	
	if((current->next == NULL) && (current->time < start_time))	//If the event to be added to the tail of event list
	{
		new->next = NULL;
		current->next = new;
	}
	else							//If new even to be added to the middel of the event list
	{
		new->next = current;
		previous->next = new;
	}
	if(previous == NULL)					//If new event to be added to the head of the event list
		root->next = new;
}

/*------------------------------------------------------------------------------*
 * Follwoing function remove the given event from the event list		*
 * start_time   -       remove the event related to the given start time	*
 * NID		-	NID of the node related to the event                    *
 *------------------------------------------------------------------------------*/
void remove_from_event_list(uint start_time, uint nid)
{
	event *current, *previous;

	if(root->next == NULL)				//Makesure the event list is not empty
	{	
		printf("Event list is already empty\n");
		exit(1);
	}

	current = root;
	previous = NULL;
	
	//Locate the corrosponding event
	while( (current->next != NULL) && (current->time < start_time) && (current->NID != nid))	
	{
		previous = current;
		current = current->next;
	}
	if(current->next == NULL)			//if its the last event
	{
		free(current);
		previous->next = NULL;
	}
	else						//If at the head or middle of the list
	{
		previous->next = current->next;
		free(current);
	}
}

/*------------------------------------------------------------------------------*
 * Process the event list starting from the first event, one event at a time	*
 * If no more events are available, print data about nodes & exit.		*
 *------------------------------------------------------------------------------*/
void process_event_list()
{
	event *next_event;
	uint x0, y0, parentx0, parenty0, parent_NID, next_route_no;
	uchar result;
	uint no_msg_delivered = 0;	//No of sucessfully delivered messages
	uint no_msg_dropped = 0;        //Nof of messages dropped
	uint no_route = 0;		//No fo routes not found
	
	if(root->next == NULL)			//If no more events to handle print the data & exit
	{
		printf("No cluster: %d\t No common key: %d\n", nodes_without_CH(0), nodes_without_CH(1));
		print_no_common_keys();		//dump no of common keys each node has with its neighbor
		while (1)			//Keep sending data until 1st packet is dropped
		{
			result = send_data();	
			if(result == 1)		//If a dropped packet
			{
				no_msg_dropped++;
				break;
			}
			else if (result == 2)	//If route doesn't exist
			{
				no_route++;
				break;
			}
			else			//If sucessful keep counting the no of messages delivered
				no_msg_delivered++;
		}

		//printf("%d\t%d\t%d\n", no_msg_delivered, no_msg_dropped, no_route);
		printf("No of messages delivered: %d\n", no_msg_delivered);
		//test_routing();
	
		calculate_circularity();	//Calsulate circularity & dump data to file
		//print_cluster_energy();		//Print current energy level of nodes to file
		print_nodes(SHOW_NODE_DATA);

		exit(0);	
	}
	else					//If the event list is not null keep processing the next event
	{		
		next_event = root->next;
		current_time = next_event->time;			//Set the current time

		x0 = (next_event->NID - 1) % NODESX;
		y0 = (next_event->NID - 1) / NODESX;
			 
		if(nodes[x0][y0].CID == 0)				//if the node in the event list is not taken by another cluster
		{
			//Add nodes to cluster starting from this node
			add_nodes_to_cluster(next_event->NID, next_event->CID, next_event->NID, next_event->depth, next_event->ttl);	

			if(nodes[x0][y0].no_child_nodes > 0)		//If able to attach at least 1 child node
			{
				nodes[x0][y0].CID = next_event->CID;		//Set the CID of the starting node		
				nodes[x0][y0].H_CID = next_event->H_CID;	//Set hierarchical CID
				nodes[x0][y0].CH_NID = next_event->NID;		//Set me as my own CH
	       			nodes[x0][y0].tree_depth = next_event->depth;	//set the depth
		       		nodes[x0][y0].parent_CH_NID = next_event->parent_CH_NID; 	

				if(next_event->parent_CH_NID != 0)			//If not the root node
				{
					nodes[x0][y0].no_routing_entries++;		//Add routing entries to the parent CH
					nodes[x0][y0].routing_table[0].valid = 3;	//Valid & parent routing entry
					nodes[x0][y0].routing_table[0].nei_H_CID = next_event->parent_H_CID;	
					nodes[x0][y0].routing_table[0].nei_NID = next_event->parent_CH_NID;	

					//Add to the parents routing table
					parent_NID = nodes[x0][y0].parent_CH_NID;
					parentx0 = (parent_NID - 1) % NODESX;
					parenty0 = (parent_NID - 1) / NODESX;
					next_route_no = nodes[parentx0][parenty0].no_routing_entries;
					nodes[parentx0][parenty0].routing_table[next_route_no].valid = 1;
					nodes[parentx0][parenty0].routing_table[next_route_no].nei_H_CID = next_event->H_CID;
					nodes[parentx0][parenty0].routing_table[next_route_no].nei_NID = next_event->NID;	
					nodes[parentx0][parenty0].no_routing_entries++;
					
				}
				//Select candidate CHs for the next round
				select_child_CHs(next_event->depth, next_event->CID, next_event->H_CID, next_event->NID);

				nodes[x0][y0].energy -= energy_to_transmit(CLUSTER_BCAST_SIZE, (float)R); 
			}
		}
		remove_from_event_list(next_event->time, next_event->NID);	//Remove the event from event list
	}
}

/*------------------------------------------------------------------------------*
 * Add child nodes to cluster.							*
 * Mark all the nodes in the communication range of the broadcasting node &	*
 * without a cluster.								*
 * nid		-	NID of the node sending the bcast			*
 * cid		-	CID of the new cluster					*
 * ch_nid	-	NID of the CH						*
 * depth	-	depth of the node sending the bcast			*
 * ttl		-	TTL of the bcast					*
 *------------------------------------------------------------------------------*/
void add_nodes_to_cluster(uint nid, uint cid, uint ch_nid, uchar depth, uchar ttl)
{
	int minx, miny, maxx, maxy, retvalue;
	uint x0, y0, no_of_neighbors, n, newx0, newy0, CH_x, CH_y, l, k, i, z;
	uint bcast_size, ack_size;
	float distance;
	key_index key_CH[MAX_Z], key_child[MAX_Z]; 
	uint neighbors[625];					//No if 1-hop neighbors. This depends on tranmissions distance
								//So no of entries in the array (2 * r + 1)^2 -1
								//Will be valid untill communication distance is 12 time grid size
	no_of_neighbors = 0;	

	if(USE_KPS == 1)
	{
		bcast_size = CLUSTER_BCAST_SIZE + KEY_INDEX_SIZE; 
		ack_size = ACK_PACKET_SIZE + KEY_INDEX_SIZE;	
	}
	else
	{
		bcast_size = CLUSTER_BCAST_SIZE; 
		ack_size = ACK_PACKET_SIZE + KEY_INDEX_SIZE;	
	}

	x0 = (nid - 1) % NODESX;
	y0 = (nid - 1) / NODESX;
	CH_x = (ch_nid - 1) % NODESX;
	CH_y = (ch_nid - 1) / NODESY;
	
	minx = x0 - (R/GRIDX) - 1;                              // 1 hop neighbors are in the range of (X0 - R) to (X0 + R)
	if (minx < 0)                                           // Stay within the grid			
		minx = 0;
	maxx = x0 + (R/GRIDX) + 1;
	if(maxx >= NODESX)
		maxx = NODESX -1;

	miny = y0 - (R/GRIDY) - 1;                              // 1 hope neighbors are in the range of (Y0 - R) tp (Y0 + R)
	if (miny < 0)                                           // Stay within the grid
		miny = 0;		
	maxy = y0 + (R/GRIDY) + 1;
	if(maxy >= NODESY)
		maxy = NODESY - 1;

	if(nodes[x0][x0].last_bcast_for == cid)			//Check whether the cluster formation message is already broadcasted by this node
		return;						//If so discard
		
	nodes[x0][y0].broadcasts++;                  	//Node is broadcastsing	
	nodes[x0][y0].last_bcast_for = cid;		//Set the CH who send the last broadcast

	//Energy consumed to transmit a message
	if(USE_KPS == 1)				//If KPS is used
	{
		nodes[x0][y0].energy -= energy_to_transmit(bcast_size, (float)R); 
		for(z = 0; z < kps_z ; z++)
			key_CH[z] = nodes[CH_x][CH_y].key_IDs[z];
	}
	else
		nodes[x0][y0].energy -= energy_to_transmit(CLUSTER_BCAST_SIZE, (float)R); 

	//Set CID for only neighboring nodes   
	for(l = miny; l <= maxy; l++)
	{
		for (k = minx; k <= maxx; k++)
		{
			//Geomatric distance between two nodes
			distance = sqrt((k - x0)*(k - x0)*(GRIDX * GRIDX) + (l - y0)*(l - y0)*(GRIDY * GRIDY));
			
			// If within communication range & if the node exist & not me
			if ((distance <= R) && (nodes[k][l].NID != 0) && (distance != 0))     
			{
				//Energy consumed to receive a message
				nodes[k][l].energy -= energy_to_receive(bcast_size);	//Energy consumed to receive a message
				
				if((MAX_TTL - ttl) < MAX_HOPS)		//If within MAX_HOPS
				{
				   	//If not a member of a cluster assigned to the current cluster
		 			if(nodes[k][l].CID == 0)
					{
						if((USE_KPS == 1) && (nodes[k][l].no_key_blocks > 0)) 	//if key predistribution scheme to be used
						{
							for(z = 0; z < kps_z ; z++)
								key_child[z] = nodes[k][l].key_IDs[z];
						
							retvalue =  block_common_key(kps_z, key_CH, key_child, kps_k, kps_p, &(nodes[k][l].common_key_ID));
							if(retvalue == 1)	//If a common key exist
							{
								nodes[k][l].CID = cid;
								nodes[k][l].CH_NID = ch_nid;
								nodes[k][l].tree_depth = depth + 1;
								nodes[CH_x][CH_y].no_child_nodes++;
						
								//Add as a node capable of forwarding the bcast
								neighbors[no_of_neighbors] = nodes[k][l].NID;
								no_of_neighbors++;

								//Energy consumed to send an ACK & energy consumed to receive ACK
								nodes[k][l].energy -= energy_to_transmit(ack_size, (float) R);	
								nodes[CH_x][CH_y].energy -= energy_to_receive(ack_size); 
							}
							else
							{
								nodes[k][l].no_key_to_CH = 1;	//unable to join cluster due to no common key
							}
						}
						else				//no key predistribution
						{
							nodes[k][l].CID = cid;
							nodes[k][l].CH_NID = ch_nid;
							nodes[k][l].tree_depth = depth + 1;
							nodes[CH_x][CH_y].no_child_nodes++;
						
							//Add as a node capable of forwarding the bcast
							neighbors[no_of_neighbors] = nodes[k][l].NID;
							no_of_neighbors++;

							nodes[k][l].energy -= energy_to_transmit(ack_size, (float) R);	
							nodes[CH_x][CH_y].energy -= energy_to_receive(ack_size); 
						}
					}
				}
				else if((MAX_TTL - ttl) < (2 * MAX_HOPS))	//if outside cluster but TTL not expired
				{
			               	if(nodes[k][l].CID == 0)		//If not already a member of a cluster	
					{
						//Add as a node capable of forwarding the bcast
						neighbors[no_of_neighbors] = nodes[k][l].NID;
						no_of_neighbors++;
					}
				}
				else if((MAX_TTL - ttl) < (2 * MAX_HOPS + 1))	//if outside cluster but TTL expired
				{
					if(nodes[k][l].CID == 0)        	// If not a memeber of any cluster
					{
						//See whether it has already been seleted as a candidate to become a CH
	                        		for( i = 0; i < no_candidate_CHs; i++)
						{
							if(candidateCHs[i] == nodes[k][l].NID)
								break;
						}
						if( i == no_candidate_CHs)
						{       
							
							if(USE_KPS == 1)
							{
								for(z = 0; z < kps_z ; z++)
								key_child[z] = nodes[k][l].key_IDs[z];
							
								//Check whether a common key exist with CH		
								retvalue =  block_common_key(kps_z, key_CH, key_child, kps_k, kps_p, &(nodes[k][l].common_key_ID));
								if(retvalue == 1)	//If a common key exist
								{
									candidateCHs[no_candidate_CHs] = nodes[k][l].NID;
									no_candidate_CHs++;

									//Energy consumed to send an ACK & energy consumed to receive ACK
									nodes[k][l].energy -= energy_to_transmit(ack_size, (float) R);	
									nodes[CH_x][CH_y].energy -= energy_to_receive(ack_size); 
								}
							}
							else
							{
								candidateCHs[no_candidate_CHs] = nodes[k][l].NID;
								no_candidate_CHs++;
								nodes[k][l].energy -= energy_to_transmit(ack_size, (float) R);	
								nodes[CH_x][CH_y].energy -= energy_to_receive(ack_size); 
							}
						}
					}
				}
				else						//If TTL expired just go back to the caller
					return;
			}
		}
	}

	for (n = 0; n < no_of_neighbors; n++)		//For each neighbor
	{
		newx0 = (neighbors[n] - 1) % NODESX;
		newy0 = (neighbors[n] - 1) / NODESX;
	
		//For each node find other 1-hop neighbors & keep forwarding cluster formation message
		ttl--;					//Decrement TTL
		depth++;				//Increment depth
		add_nodes_to_cluster(neighbors[n], cid, ch_nid, depth, ttl); //recursively call the function

		//reset values
		depth--;				//Decrement depth
		ttl++;					//Increment ttl
	}
}


/*------------------------------------------------------------------------------*
 * This function selecte candidate CHs from a list of nodes that is set 	*
 * during the add_nodes_to_cluster phase.					*
 * depth	- 	depth of the new clsuter in the cluster tree		*
 * parent_cid	-	CID of the parent CH					*
 * parent_h_cid	-	Hierarchical CID of the parent CH			*
 * parent_ch_nid-	NID of the parent CH					*
 *------------------------------------------------------------------------------*/
void select_child_CHs(uchar depth, uint parent_cid, Hie_CID parent_h_cid, uint parent_ch_nid)
{
	uint no_new_chs, next_ch, delay_next_cluster, j, l, new_time, retvalue;
	uint parent_x0, parent_y0, child_x0, child_y0;
	uint selected_ch_list[10];		//list of already selected candidate CHs
	uchar no_selected_chs = 0;		//no of already selected candidate CHs
	uchar z, loops;				//no of key indexes in a node
	Hie_CID new_h_cid;			//Hierarchical CID of the new cluster	
	key_index key_parent[MAX_Z], key_child[MAX_Z]; 
	
	if(no_candidate_CHs == 0)		//If no nodes are there to be elected as CHs return function
		return;	
	
	if(depth < 30)				//If no of candidate CHs for each round is given in the header file
		no_new_chs= no_new_nodes[depth];		
	else					//if not use the last no given in the header file
		no_new_chs = no_new_nodes[29];

	if(no_new_chs > no_candidate_CHs)	//If no of possible CHs are less than what is given in the header file
		no_new_chs = no_candidate_CHs;	//Generate the maximum possible no

	if(USE_KPS == 1)					//if key predistribution scheme to be used
	{	
		parent_x0 = (parent_ch_nid - 1) % NODESX;
		parent_y0 = (parent_ch_nid - 1) / NODESX;
		for(z = 0; z < kps_z ; z++)
			key_parent[z] = nodes[parent_x0][parent_y0].key_IDs[z];
	}

	for(j = 0; j < no_new_chs; j++)			//Select given no of candidate neighbors as CHs
	{
		while(1)
		{
			next_ch = rand() % no_candidate_CHs; 		//Randomly select one of the nodes to be the CH
			
			for(l = 0 ; l < no_selected_chs ; l++)		//Check whether it has been already selected
			{
				if(selected_ch_list[l] == candidateCHs[next_ch])	//If a match found don't selecte again
					break;	
			}
			if(l != no_selected_chs)			//If a match is found in the array skip
				continue;
			
			//if key predistribution scheme to be used don't select nodes with a common key with the parent CH
			if(USE_KPS == 1)				
			{
				child_x0 = (candidateCHs[next_ch] - 1) % NODESX;
				child_y0 = (candidateCHs[next_ch] - 1) / NODESX;
				for(z = 0; z < kps_z ; z++)
					key_child[z] = nodes[child_x0][child_y0].key_IDs[z];
						
				retvalue =  block_common_key(kps_z, key_parent, key_child, kps_k, kps_p, &(nodes[child_x0][child_y0]. common_key_ID));
				if(retvalue == 0)	//If no common key exist skip the node
				{
					loops++;	//increment loop counter when common node is not found	
					//200 is just an aribiatory value just to prevent the loop form becoming infinite
					if(loops == 200) //Just keep looping until a possble node is found or until loop count exceeds.
						break;
					else
						continue;
				}
			}				
			//Keep track of the selected one to prevent it being selected again
			selected_ch_list[no_selected_chs] = candidateCHs[next_ch];
                        no_selected_chs++;
				
			nextCID++;					//generate the next CID

			if(depth < 30)
				delay_next_cluster = delay[depth][j];
			else
				delay_next_cluster = delay[29][j];
			
			//Set the timeing such that tree is formed using the Breadth-first tree formation
			new_time = last_event_time + delay_next_cluster + CLUSTER_DELAY;

			//Add to the event list
			new_h_cid = generate_CID(parent_h_cid, j, ( depth + 1));
	 		add_to_event_list(new_time, candidateCHs[next_ch], nextCID, new_h_cid, (depth + 1), MAX_TTL, parent_cid, parent_ch_nid, parent_h_cid);
			break;
		}
	}
	no_candidate_CHs = 0;
}

/*------------------------------------------------------------------------------*
 * Follwing function calculates the Maximum Achievable Circularity (MAC) for    *
 * each cluster & print circularity to the circularity.txt file                 *
 * This functions works only up to 1-hop clusters. Can be extened to 4-hops by  *
 * adding appropriate codes                                                     *
 *------------------------------------------------------------------------------*/
void calculate_circularity()
{
	int i, j, x0, y0, minx, miny, maxx, maxy, l, k, retvalue;
	float length, in_cluster, outof_cluster, circularity;
	uint nb_list[3][500];
	uint level0, level1, level2, n, m, p, q;
	
      	FILE *circlefd;
        circlefd = fopen(CIRCLEFILE, "w");
	
        for (i = 0; i < NODESX ; i++)	      
	{
		for(j = 0 ; j < NODESY ; j++)
		{
			//Make sure that the node is a CH
			if((nodes[i][j].NID == nodes[i][j].CH_NID) && (nodes[i][j].NID != 0))
			{
				in_cluster = 0;				//No of nodes in side cluster
				outof_cluster = 0;			//No of nodes out side cluster
				level0 = level1 = level2 = 0;	//Check the hope for hop-head & multi-hop clustering
      			
				x0 = (nodes[i][j].NID - 1) % NODESX;
				y0 = (nodes[i][j].NID - 1) / NODESX;

				minx = x0 - (R/GRIDX);          // 1 hop neighbors are in the range of (X0 - R) to (X0 + R)
			        if (minx < 0)                           // Stay within the grid
	                		minx = 0;
			        maxx = x0 + (R/GRIDX);
			        if(maxx >= NODESX)
	                		maxx = NODESX -1;

			        miny = y0 - (R/GRIDY);             // 1 hope neighbors are in the range of (Y0 - R) tp (Y0 + R)
			        if (miny < 0)                              // Stay within the grid
	                		miny = 0;
				maxy = y0 + (R/GRIDY);
			        if(maxy >= NODESY)
	                		maxy = NODESY - 1;
	
        			for(l = miny; l <= maxy; l++)		
				{
					for (k = minx; k <= maxx; k++)
					{
						length = sqrt((k - x0)*(k-x0)*(GRIDX * GRIDX) + (l - y0)*(l-y0)*(GRIDY * GRIDY));
						if((length <=  R) && (nodes[k][l].NID != 0))     // If within communication range
						{					
							if(nodes[k][l].CID == nodes[i][j].CID)
								in_cluster++;
							else
								outof_cluster++;
							nb_list[0][level0] = nodes[k][l].NID;
							level0++;		
						}
					}
				}
				if( (MAX_HOPS > 1) && (level0 != 0))
				{
					for(n=0; n < level0 ; n++)
					{
						x0 = (nb_list[0][n] - 1) % NODESX;
						y0 = (nb_list[0][n] - 1) / NODESX;
						
						minx = x0 - (R/GRIDX);  
						if (minx < 0)                           // Stay within the grid
							minx = 0;
						maxx = x0 + (R/GRIDX);
				 		if(maxx >= NODESX)
		   					maxx = NODESX -1;
												                                
						miny = y0 - (R/GRIDY);  
						if (miny < 0)                          // Stay within the grid
							miny = 0;
						maxy = y0 + (R/GRIDY);
						if(maxy >= NODESY)
							maxy = NODESY - 1;																				
		        			for(l = miny; l <= maxy; l++)
						{
							for (k = minx; k <= maxx; k++)
							{
								length = sqrt((k-x0)*(k-x0)*(GRIDX * GRIDX) + (l-y0)*(l-y0)*(GRIDY * GRIDY));
								if((length <=  R) && (nodes[k][l].NID != 0))  
								{
									for(m = 0; m < level0; m++)
									{
										if(nb_list[0][m] == nodes[k][l].NID)
											break;
									}
									for( p = 0; p < level1; p++)
									{
										if(nb_list[1][p] == nodes[k][l].NID)
											break;
									}
									if((m == level0) && (p == level1))
									{
										if(nodes[k][l].CID == nodes[i][j].CID)
											in_cluster++;
										else
											outof_cluster++;
										nb_list[1][level1] = nodes[k][l].NID;
										level1++;		
									}
								}
							}
						}
					}
				}
				if( (MAX_HOPS > 2) && (level1 != 0))
                                {
                                        for(n=0; n < level1 ; n++)
                                        {
                                                x0 = (nb_list[1][n] - 1) % NODESX;
                                                y0 = (nb_list[1][n] - 1) / NODESX;

                                                minx = x0 - (R/GRIDX);
                                                if (minx < 0)                           // Stay within the grid
                                                        minx = 0;
                                                maxx = x0 + (R/GRIDX);
                                                if(maxx >= NODESX)
                                                        maxx = NODESX -1;

                                                miny = y0 - (R/GRIDY);
                                                if (miny < 0)                          // Stay within the grid
                                                        miny = 0;
                                                maxy = y0 + (R/GRIDY);
                                                if(maxy >= NODESY)
                                                        maxy = NODESY - 1;

						for(l = miny; l <= maxy; l++)
                                                {
                                                        for (k = minx; k <= maxx; k++)
                                                        {
                                                                length = sqrt((k-x0)*(k-x0)*(GRIDX * GRIDX) + (l-y0)*(l-y0)*(GRIDY * GRIDY));
                                                                if((length <=  R) && (nodes[k][l].NID != 0))
                                                                {
                                                                        for(m = 0; m < level0; m++)
                                                                        {
                                                                                if(nb_list[0][m] == nodes[k][l].NID)
                                                                                        break;
                                                                        }
                                                                        for( p = 0; p < level1; p++)
                                                                        {
                                                                                if(nb_list[1][p] == nodes[k][l].NID)
                                                                                        break;
                                                                        }
                                                                        for(q= 0; q < level2; q++)
                                                                        {
                                                                                if(nb_list[2][q] == nodes[k][l].NID)
                                                                                        break;
                                                                        }

                                                                        if((m == level0) && (p == level1) && (q == level2))
                                                                        {
                                                                                if(nodes[k][l].CID == nodes[i][j].CID)
                                                                                        in_cluster++;
                                                                                else
                                                                                        outof_cluster++;
                                                                                nb_list[2][level2] = nodes[k][l].NID;
                                                                                level2++;
                                                                        }
                                                                }
                                                        }
                                                }

                                        }

                                }

				circularity = (in_cluster/(in_cluster + outof_cluster)) * 100;
				retvalue = sprintf(msg, "%f\n", circularity);
				fputs(msg, circlefd);
			}
		}
	}		
	fclose(circlefd);	//Close file
}

/*------------------------------------------------------------------------------*
 * Following function generates the hierarchical cluster ID based on the parent	*
 * cluster ID & node depth. CIDs are represented in the following manner	*
 *     +------------------------+----+------------------+			*
 *     |  ......|Child2|Child1|Parent|  Depth (5- bits)	|			*
 *     +------------------------+----+------------------+			*
 *     Depth - 5 bits - Can represent upto depth of 32 levels in the tree 	*
 *     Branching factor (B) - up to 8 bracnhes (3 bits to represent each brancg	*
 *     number. 									*
 *     Child ID = Parent ID + (Branch No << ((Depth -1) * B)+ 5) + Depth 	*
 *     With 64-bit CID up to 19 level cluster tree can be represented		*
 *     However there are no 64-bit integer data structures. So an array of two	*
 *     32-bit integers are used.						*  
 *------------------------------------------------------------------------------*/
Hie_CID generate_CID(Hie_CID parent_ID, int child_no, int depth)
{
	uint tmp_id0, tmp_id1, tmp_id2,  tmp_parent_ID[3], tmp_depth;
	Hie_CID new_H_CID;
	
	tmp_parent_ID[0] = parent_ID.id[0];
	tmp_parent_ID[1] = parent_ID.id[1];
	tmp_parent_ID[2] = parent_ID.id[2];
	
	tmp_depth = tmp_parent_ID[2] & 31;			//Extract the depth 31 = 11111 in binary

	if ((tmp_depth + 1) != depth)				//If child depth != parent depth + 1
	{
		printf("Child depth should be equal to parent depth + 1\n");
		exit(0);
	}
	
	if(depth <= 9)					//up to depth of 9 can be represented by 32-bits (3 * 9 + 5)
	{
		tmp_id2 = child_no;
		tmp_id2 = tmp_id2 << (((depth -1) * 3) + 5); 
		tmp_id2 += tmp_parent_ID[2];
		tmp_id2++;				//Increment depth by 1 
		tmp_id1 = 0;				//The the last (MSB) 32-bits zero
		tmp_id0 = 0;				//The the last (MSB) 32-bits zero
	}
	else if((depth > 9) && (depth < 20)) 					//If within 10 to 19
	{
		tmp_id2 = tmp_parent_ID[2];
		tmp_id2++;				//Increment depth
		tmp_id1 = child_no;			//Move to the other part of the CID and set it
		tmp_id1 = tmp_id1 << ((depth - 10) * 3);//Shift the new no
		tmp_id1 += tmp_parent_ID[1];		//Add the MSB part of parent
		tmp_id0 = 0;				//The the last (MSB) 32-bits zero
	}
	else 				 					//If not
	{
		tmp_id2 = tmp_parent_ID[2];
		tmp_id2++;				//Increment depth
		tmp_id1 = tmp_parent_ID[1];
		tmp_id0 = child_no;			//Move to the other part of the CID and set it
		tmp_id0 = tmp_id0 << ((depth - 20) * 3);//Shift the new no
		tmp_id0 += tmp_parent_ID[0];		//Add the MSB part of parent
	}
	new_H_CID.id[0] = tmp_id0;
	new_H_CID.id[1] = tmp_id1;
	new_H_CID.id[2] = tmp_id2;

	return new_H_CID;					//Return new CID
}


/*------------------------------------------------------------------------------*
 * Follwoing function returns the NID of the next hop to forward the message	*
 * If it is for the current clusters same NID is returned			*
 * If its for a neighboring cluster its NID is returned				*
 * Otherwice the NID of the next hop is returned based on the entries in the 	*
 * routing table. 								*
 * If no suitable next hop can't be found 0 is returned				*
 *------------------------------------------------------------------------------*/
int next_hop(Hie_CID dest_add, int current_NID, int sender_NID)
{
	uint x0, y0, current_depth, dest_depth, min_depth, i, j, tmp_current_add, tmp_dest_add;
	nei_status  neighbors[MAXROUTES];			//Hold the hop count for each router table entry
	int no_routing_entries;		
			
	i = 0;
	x0 = (current_NID - 1) % NODESX;
	y0 = (current_NID - 1) / NODESX;
	
	//Check whether the destination is my address
	if((nodes[x0][y0].H_CID.id[0] == dest_add.id[0]) && (nodes[x0][y0].H_CID.id[1] == dest_add.id[1]) && (nodes[x0][y0].H_CID.id[2] == dest_add.id[2]))
		return current_NID;

	no_routing_entries = nodes[x0][y0].no_routing_entries;

	for(j = 0; j < no_routing_entries; j++)
	{
		if((nodes[x0][y0].routing_table[j].valid & 1) == 0) 	//If route is invalid discard. Validity is indicated by 1st bit
		{
			neighbors[j].hops = 255;			//Set as unreachable	
			continue;
		}

		//Check whether the destination is the neighbors address
		if((nodes[x0][y0].routing_table[j].nei_H_CID.id[0] == dest_add.id[0]) && (nodes[x0][y0].routing_table[j].nei_H_CID.id[1] == dest_add.id[1]) && (nodes[x0][y0].routing_table[j].nei_H_CID.id[2] == dest_add.id[2]))
			return nodes[x0][y0].routing_table[j].nei_NID;

		current_depth = nodes[x0][y0].routing_table[j].nei_H_CID.id[2] & 31;		//31 = 11111 in binary
		dest_depth = dest_add.id[2] & 31;
		
		if((current_depth > 29) || (dest_depth > 29))		//If depth > 29 Hierarchical Address will overflow
			return 0;

		if(current_depth > dest_depth)				//Find minimum depth 
			min_depth = dest_depth;
		else
			min_depth = current_depth;

		for(i = 0; i < min_depth ; i++)
		{
			if(i < 9)					//Check only the first 32-bits of the address
			{	
				tmp_current_add = (nodes[x0][y0].routing_table[j].nei_H_CID.id[2] >> (5 + i * 3));	//Remove the depth info from address 
				tmp_dest_add = (dest_add.id[2] >> (5 + i * 3));
				tmp_current_add = tmp_current_add & 7;	// 7 = 111 in binary. Extract only the branch no for level i+1
				tmp_dest_add = tmp_dest_add & 7;
				if(tmp_current_add != tmp_dest_add)	//If a none matching branch is found	
					break;
			}
			else if(( i >= 9) && (i < 19))			//Check the remanining bits
			{
				tmp_current_add = (nodes[x0][y0].routing_table[j].nei_H_CID.id[1] >> ((i - 9) * 3));	//Remove other part of address
				tmp_dest_add = (dest_add.id[1] >> ((i - 9) * 3));
				tmp_current_add = tmp_current_add & 7;	// 7 = 111 in binary. Extract only the branch no for level i+1
				tmp_dest_add = tmp_dest_add & 7;	
				if(tmp_current_add != tmp_dest_add)	//If a none matching branch is found	
					break;
			}
			else
			{
				tmp_current_add = (nodes[x0][y0].routing_table[j].nei_H_CID.id[0] >> ((i - 19) * 3));	//Remove other part of address 
				tmp_dest_add = (dest_add.id[0] >> ((i - 19) * 3));
				tmp_current_add = tmp_current_add & 7;	// 7 = 111 in binary. Extract only the branch no for level i+1
				tmp_dest_add = tmp_dest_add & 7;	
				if(tmp_current_add != tmp_dest_add)	//If a none matching branch is found	
					break;
			}
		}
		neighbors[j].hops = (current_depth - i) + (dest_depth - i) + 1;	//add the hops up to the common branching point. 
									//+1 because therese are neighbors & 1-hop is need to reach it

		neighbors[j].nei_NID = nodes[x0][y0].routing_table[j].nei_NID;

		if((nodes[x0][y0].routing_table[j].valid & 2) == 2) 	//If route is learned from parent CH
		{
			neighbors[j].parent = 1;
			neighbors[j].child = 0;
			neighbors[j].neigh = 0;					
		}
		else if((nodes[x0][y0].routing_table[j].valid & 4) == 4) 	//If route is learned from neighbor CH
		{
			neighbors[j].neigh = 1;					
			neighbors[j].child = 0;
			neighbors[j].parent = 0;
		}
		else						//If learn from child CH
		{
			neighbors[j].child = 1;
			neighbors[j].neigh = 0;					
			neighbors[j].parent = 0;
		}
	}

	bubble_sort(neighbors, no_routing_entries);		// Sort based on hop count
     
	if(neighbors[0].hops == 255)
		return 0;					//No matching next hop found
	else if (neighbors[0].nei_NID != sender_NID)		//If next hop is not same as the send of the message allow that
		return neighbors[0].nei_NID;
	else
	{
		for(j = 1 ; j < no_routing_entries; j++)                //Find a node which is not energy constrained
                {
                	if(( neighbors[j].hops != 255 ) && (neighbors[j].nei_NID != sender_NID)) //If found stop
                        	return neighbors[j].nei_NID;
		}
		if( j == no_routing_entries)		//No possible next node
			return 0;
	}
}

void test_routing()
{
	int x0, y0, tmp_node, count;
	int tmp_nodes[5];
		
	count = 0;
	while(1)
	{
    		tmp_node = rand() % 401;
    		x0 = (tmp_node - 1) % NODESX;	    
		y0 = (tmp_node - 1) / NODESX;
		if(nodes[x0][y0].NID == 0)
			continue;
		if(nodes[x0][y0].NID != nodes[x0][y0].CH_NID)
			continue;
		tmp_nodes[count] = nodes[x0][y0].NID;
		count++;
		if(count == 5)
			break;
	}
	
    	x0 = (tmp_nodes[1] - 1) % NODESX;	    
	y0 = (tmp_nodes[1] - 1) / NODESX;
/*
	printf("Source: %d\t Dest: %d Next Hop: %d\n", tmp_nodes[0], tmp_nodes[1], next_hop(nodes[x0][y0].H_CID, tmp_nodes[0]));
    	x0 = (tmp_nodes[2] - 1) % NODESX;	    
	y0 = (tmp_nodes[2] - 1) / NODESX;
	printf("Source: %d\t Dest: %d Next Hop: %d\n", tmp_nodes[0], tmp_nodes[2], next_hop(nodes[x0][y0].H_CID, tmp_nodes[0]));
    	x0 = (tmp_nodes[3] - 1) % NODESX;	    
	y0 = (tmp_nodes[3] - 1) / NODESX;
	printf("Source: %d\t Dest: %d Next Hop: %d\n", tmp_nodes[0], tmp_nodes[3], next_hop(nodes[x0][y0].H_CID, tmp_nodes[0]));
    	x0 = (tmp_nodes[2] - 1) % NODESX;	    
	y0 = (tmp_nodes[2] - 1) / NODESX;
	printf("Source: %d\t Dest: %d Next Hop: %d\n", tmp_nodes[1], tmp_nodes[2], next_hop(nodes[x0][y0].H_CID, tmp_nodes[1]));
    	x0 = (tmp_nodes[3] - 1) % NODESX;	    
	y0 = (tmp_nodes[3] - 1) / NODESX;
	printf("Source: %d\t Dest: %d Next Hop: %d\n", tmp_nodes[1], tmp_nodes[3], next_hop(nodes[x0][y0].H_CID, tmp_nodes[1]));
    	x0 = (tmp_nodes[3] - 1) % NODESX;	    
	y0 = (tmp_nodes[3] - 1) / NODESX;
	printf("Source: %d\t Dest: %d Next Hop: %d\n", tmp_nodes[2], tmp_nodes[3], next_hop(nodes[x0][y0].H_CID, tmp_nodes[2]));
    	x0 = (tmp_nodes[4] - 1) % NODESX;	    
	y0 = (tmp_nodes[4] - 1) / NODESX;
	printf("Source: %d\t Dest: %d Next Hop: %d\n", tmp_nodes[3], tmp_nodes[4], next_hop(nodes[x0][y0].H_CID, tmp_nodes[3]));
    	x0 = (tmp_nodes[2] - 1) % NODESX;	    
	y0 = (tmp_nodes[2] - 1) / NODESX;
	printf("Source: %d\t Dest: %d Next Hop: %d\n", tmp_nodes[4], tmp_nodes[2], next_hop(nodes[x0][y0].H_CID, tmp_nodes[4]));
*/
}

/*------------------------------------------------------------------------------*
 * Following functions randomly generate a socue and a destination node	& then	*
 * try to deliver a message between those 2 nodes. Function returns:		*
 * 0 - On sucess								*
 * 1 - when not enough energy to deliver the packet				*
 * 2 - When route to destination is not found 					*
 *------------------------------------------------------------------------------*/
unsigned char send_data()
{
        uint s_x, s_y, d_x, d_y, CH_NID, source_CH_x, source_CH_y, dest_CH_x, dest_CH_y, receiver_x, receiver_y;
	uint neighbor_to_forward;
	uint current_x, current_y, current_NID;
	uint msg_send_by;			//No that send the message
	
	while (1)				//Generate source node
	{
		s_x = rand() % NODESX;
		s_y = rand() % NODESY;
		if((nodes[s_x][s_y].NID == 0) ||(nodes[s_x][s_y].CID == 0))	//Make sure node is available & in a cluster
			continue;
		else if(nodes[s_x][s_y].node_dead == 1)				//Source node is dead find another node
			continue;
		else				//Form the source info of the data packet
		{
			data_packet.source_NID = nodes[s_x][s_y].NID;
			CH_NID = nodes[s_x][s_y].CH_NID;
    			source_CH_x = (CH_NID - 1) % NODESX;	    
			source_CH_y = (CH_NID - 1) / NODESX;
			data_packet.source_H_CID = nodes[source_CH_x][source_CH_y].H_CID;
			//data_packet.source_H_CID.id[0] = nodes[source_CH_x][source_CH_y].H_CID.id[0];
			//data_packet.source_H_CID.id[1] = nodes[source_CH_x][source_CH_y].H_CID.id[1];
			//data_packet.source_H_CID.id[2] = nodes[source_CH_x][source_CH_y].H_CID.id[2];
			break;
		}
	}	
	while (1)				//Generate destination node
	{
		d_x = rand() % NODESX;
		d_y = rand() % NODESY;
		if((nodes[d_x][d_y].NID == 0) || (nodes[d_x][d_y].CID == 0))	//Make sure node is available & a member of a cluster
			continue;
		else if(nodes[d_x][d_y].node_dead == 1)				//Source node is dead find another node
			continue;
		else if((s_x == d_x) && (s_y == d_y))				//if both source & destination is equal find another
			continue; 
		else				//Form the destination info of the data packet
		{
			data_packet.dest_NID = nodes[d_x][d_y].NID;
			CH_NID = nodes[d_x][d_y].CH_NID;
    			dest_CH_x = (CH_NID - 1) % NODESX;	    
			dest_CH_y = (CH_NID - 1) / NODESX;
			data_packet.dest_H_CID = nodes[dest_CH_x][dest_CH_y].H_CID;
			//data_packet.dest_H_CID.id[0] = nodes[dest_CH_x][dest_CH_y].H_CID.id[0];
			//data_packet.dest_H_CID.id[1] = nodes[dest_CH_x][dest_CH_y].H_CID.id[1];
			//data_packet.dest_H_CID.id[2] = nodes[dest_CH_x][dest_CH_y].H_CID.id[2];
			break;
		}
	}

	if(data_packet.source_NID != nodes[s_x][s_x].CH_NID)	//If the souce is not the CH. Then forward the message to CH
	{
		nodes[s_x][s_y].energy -= energy_to_transmit(DATA_PACKET_SIZE , (float)R);	
		if(nodes[s_x][s_y].energy < 0)			//If not enough energy
		{
			nodes[s_x][s_y].node_dead = 1;		//Mark node as dead
			return 1;
		}

		if(nodes[source_CH_x][source_CH_y].node_dead == 1)		//If the rceiving CH is dead drop message
			return 1;
		
		nodes[source_CH_x][source_CH_y].energy -= energy_to_receive(DATA_PACKET_SIZE);
		if(nodes[source_CH_x][source_CH_y].energy < 0)			//If the receiver doesn't have enough energy to receive
		{
			nodes[source_CH_x][source_CH_y].node_dead = 1;		//Mark node as dead
			return 1;
		}

		msg_send_by = nodes[s_x][s_y].NID;
		nodes[s_x][s_y].no_msg_forward++;		//Count the no of messages forwarded
	}
	
	current_NID = nodes[s_x][s_y].CH_NID;			// Start forwarding from the CH

	//Loop until destination CH is found
	while(1)
	{
		neighbor_to_forward =  next_hop(data_packet.dest_H_CID, current_NID, msg_send_by);
		if(neighbor_to_forward == current_NID)
			break;
		else if (neighbor_to_forward == 0)
			return 2;				//No troute to destination
		else
		{	
			current_x = (current_NID - 1) % NODESX;	    
			current_y = (current_NID - 1) / NODESX;
			
			if(nodes[current_x][current_y].node_dead == 1)	//Node is dead can't forward the message
			{
				return 1;
			}

			//Reduce energy to transmit;
			nodes[current_x][current_y].energy -=  energy_to_transmit(DATA_PACKET_SIZE, (float) ( CH_CH_R_FACT * R));
			
			if(nodes[current_x][current_y].energy < 0)
			{
				nodes[current_x][current_y].node_dead = 1;           	//Mark node as dead
				return 1;						//Not enough energy
			}			     

			if(nodes[current_x][current_y].sent_inform_neighbor != 1)	//If the message has alread not being send
			{
				inform_neighbors(current_NID, (CH_CH_R_FACT * R));	//Add to neighbors routing table when transmitting
				nodes[current_x][current_y].sent_inform_neighbor = 1;	//Mark as message is send
			}

			receiver_x = (neighbor_to_forward - 1) % NODESX;	    
			receiver_y = (neighbor_to_forward - 1) / NODESX;

                        if(nodes[receiver_x][receiver_y].node_dead == 1)       //Node is dead can't receive the message
                                return 1;

			//Reduce energy to receive
			nodes[receiver_x][receiver_y].energy -= energy_to_receive(DATA_PACKET_SIZE);

			if(nodes[receiver_x][receiver_y].energy < 0)	//Not enough energy
			{
				nodes[receiver_x][receiver_y].node_dead = 1;           	//Mark node as dead
				return 1;
			}
			msg_send_by = nodes[current_x][current_y].NID;
			nodes[current_x][current_y].no_msg_forward++;	//Count the no of messages forwarded
			current_NID = neighbor_to_forward ;	//Forward packet to neighbor. Neighbor becomes to current node 
		}
	}

	if(current_NID != data_packet.dest_NID)
	{
		//current node x, y values are cacluated in the previous loop
		
		if(nodes[receiver_x][receiver_y].node_dead == 1)       //Node is dead can't forward the message
                	return 1;

		nodes[receiver_x][receiver_y].energy -= energy_to_transmit(DATA_PACKET_SIZE, (float)R);	
		if(nodes[receiver_x][receiver_y].energy < 0)	//Not enough energy
		{	
			nodes[receiver_x][receiver_y].node_dead = 1;           	//Mark node as dead
			return 1; 
		}		

		nodes[d_x][d_y].energy -= energy_to_receive(DATA_PACKET_SIZE);
		if(nodes[dest_CH_x][dest_CH_y].energy < 0)	//Not enough energy
		{
			nodes[d_x][d_y].node_dead = 1;             //Mark node as dead
			return 1;
		}
		nodes[receiver_x][receiver_y].no_msg_forward++;   //Count the no of messages forwarded
	}
	return 0;	//Packet is sucessfully delivered
}


/*------------------------------------------------------------------------------*
 * Following function add the neighboring cluster information while a CH tranmit*
 * a message. If the information about neighboring cluster is already in the 	*
 * routing table it will not be added						*
 * NID - NID of the node transmitting						*
 * range - range of the transmitted message					*	 
 *------------------------------------------------------------------------------*/
void inform_neighbors(int NID, int range)
{
        int x0, y0, minx, miny, maxx, maxy, l, k, i, z, retvalue;     
	float length;
	key_index key_sender[MAX_Z], key_receiver[MAX_Z]; 
	
	x0 = (NID - 1) % NODESX;
	y0 = (NID - 1) / NODESX;
								
	minx = x0 - (range/GRIDX) - 1;		// 1 hop neighbors are in the range of (X0 - R) to (X0 + R)
	if (minx < 0)                           // Stay within the grid
		minx = 0;

	maxx = x0 + (range/GRIDX) + 1;
	if(maxx >= NODESX)
		maxx = NODESX -1;
	
	miny = y0 - (range/GRIDY) - 1;          // 1 hope neighbors are in the range of (Y0 - R) tp (Y0 + R)
	if (miny < 0)                           // Stay within the grid
		miny = 0;

	maxy = y0 + (range/GRIDY) + 1;
	if(maxy >= NODESY)
		maxy = NODESY - 1;
	
	if(USE_KPS == 1)			//if key predistribution scheme to be used
	{	
		for(z = 0; z < kps_z ; z++)
			key_sender[z] = nodes[x0][y0].key_IDs[z];
	}

        for(l = miny; l <= maxy; l++)	
      	{
		for (k = minx; k <= maxx; k++)
		{
			length = sqrt((k - x0)*(k - x0)*(GRIDX * GRIDX) + (l - y0)*(l - y0)*(GRIDY * GRIDY));
			
			// If within communication range, if node exists, if node is a CH
    			if ((length <= range) && (nodes[k][l].NID != 0) && (nodes[k][l].NID == nodes[k][l].CH_NID) && (length != 0))
			{
				for(i = 0; i < nodes[k][l].no_routing_entries; i++)	//Check whther the entry already exit
				{
					if(nodes[k][l].routing_table[i].nei_NID == NID )
						break;
				}
				if( i != nodes[k][l].no_routing_entries)	//if NID already in routing table skip
					continue;
				else						//Else add to the routing table
				{
					if(USE_KPS == 1)			//if key predistribution to be used
					{
						for(z = 0; z < kps_z ; z++)
							key_receiver[z] = nodes[k][l].key_IDs[z];
							
						retvalue =  block_common_key(kps_z, key_sender, key_receiver, kps_k, kps_p, &(nodes[k][l].common_key_ID));
						if(retvalue == 1)	//If a common key exist add to routing table else just skip
						{
							nodes[k][l].routing_table[i].nei_NID = NID;
							nodes[k][l].routing_table[i].nei_H_CID = nodes[x0][y0].H_CID;
							nodes[k][l].routing_table[i].valid = 5;		//Add as an entry learn from neighbor, 5 = 101
							nodes[k][l].no_routing_entries++;
						}
					}
					else
					{
			
						nodes[k][l].routing_table[i].nei_NID = NID;
						nodes[k][l].routing_table[i].nei_H_CID = nodes[x0][y0].H_CID;
						nodes[k][l].routing_table[i].valid = 5;		//Add as an entry learn from neighbor, 5 = 101
						nodes[k][l].no_routing_entries++;
					}
				}
			}
		}
	}
}


/*------------------------------------------------------------------------------*
 * This function either printer the node status on the terminal or print node   *
 * data to a text file named nodes.txt                                          *
 * symbols:                                                                     *
 * '.' - Indicate grid points with nodes                                        *
 * 'o' - Indicate nodes with a CH                                               *
 * '?' - Indicare nodes clusters that don't have a represnetable symbol         *
 * Cluster symbol followed by a . indicate CHs                                  *
 *------------------------------------------------------------------------------*/
void print_nodes(uchar pnt_console)
{
        int i, j, retvalue;
	FILE *nodefd;
	nodefd = fopen(NODEFILE, "w");

	//Following code write node data to the text file	
	for (i = 0; i < NODESY ; i++)
	{
		for(j = 0 ; j < NODESX ; j++)
		{
			if(nodes[j][i].NID != 0)
			{
				retvalue = sprintf(msg, "%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\n", nodes[j][i].NID, nodes[j][i].CID, nodes[j][i].broadcasts, nodes[j][i].CH_NID, nodes[j][i].no_child_nodes, nodes[j][i].tree_depth, nodes[j][i].no_key_to_CH, nodes[j][i].no_msg_forward);
				fputs(msg, nodefd);
			}
		}
	}
	fclose(nodefd);
	
	if(pnt_console == 1)
	{
		for (i = 0; i < NODESY ; i++)
		{
			for(j = 0 ; j < NODESX ; j++)
			{
				if(nodes[j][i].NID == 0)	
					printf(". ");
				else
				{
					printf("%c", CID_to_symbol_mapping(nodes[j][i].CID));
				
					if(nodes[j][i].NID == nodes[j][i].CH_NID)
						printf(".");
					else
						printf(" ");
				}
			}	
			printf("\n");	
		}
	}
}

void print_cluster_energy()
{
        int i, j, retvalue;
	FILE *energyfd;        
	energyfd = fopen(ENERGYFILE, "w");

	//Following code write node energy to the text file
	for (i = 0; i < NODESY ; i++)
	{
		for(j = 0 ; j < NODESX ; j++)
		{
			if(nodes[j][i].NID != 0)
			{
				retvalue = sprintf(msg, "%f\n", nodes[j][i].energy);
				fputs(msg, energyfd);
			}
		}
	}
	fclose(energyfd);
}

/*------------------------------------------------------------------------------*
 * Following function sort the neighbor routing entries in the increasing order	*
 * based on hop count. If 2 neighbors has the same hop count they are swapped	*
 * by tossing a coin. This prevents the same neighbor being slected again & 	*
 * again when they have the same hop count - allow load to be disteributed	*
 * neigh - pointer to the neighbor routing array				*
 * n - no of entries in the array
 *------------------------------------------------------------------------------*/
void bubble_sort(nei_status *neigh, int n)
{
	char swapped;
	int i;
	nei_status tmp;

	do
	{
		swapped = 0;
		for(i = 0; i < (n-1); i++)
		{
			if(neigh[i].hops > neigh[i+1].hops)
			{
				swapped = 1;
				tmp = neigh[i+1];
				neigh[i+1] = neigh[i];
				neigh[i] = tmp;
			}
			else if(neigh[i].hops == neigh[i+1].hops)
			{
 				if((rand() % 2) == 0 ) //Toss a coin & decide wich entry to use if 0 change if 1 keep  previous entry
                        	{
					tmp = neigh[i+1];
                                	neigh[i+1] = neigh[i];
                                	neigh[i] = tmp;
                        	}
			}
		}
	}
	while (swapped == 1);
}

/*------------------------------------------------------------------------------*
 * Following function assign key indexes for node				*
 * It assigns only the key index not individual key IDs				*
 *------------------------------------------------------------------------------*/
void assign_key_id()
{
	FILE *key_fd;			//File descripter
	char msg[100];			//Hold the string read from the file
	int retvalue;
	uint i, l, m, kps_v, kps_b, rnd_key, kps_n;
	uint tmp_key_id[8];		//Holds the key id
	key_index *key_list;		//List of key indexes

	key_fd = fopen(KEYFILE, "r");	//Open file as read only
	if(key_fd == NULL)		//If unable to open file
	{
		perror("Error- Key ID file: ");
		exit(1);
	}

	fgets(msg, 100, key_fd);	//Read 1st line from key_id file
	retvalue = sscanf(msg, "%d %d %d %d %d", &kps_v, &kps_b, &kps_p, &kps_k, &kps_z);

	if(retvalue < 1)		//If unable to read
	{
		printf("Unable to read key id file. Terminating....\n");
		exit(1);
	} 
	if(kps_b < no_of_nodes)		//If no of key blocks in file is < no of sensor nodes
	{
		printf("No of keys is less than no of nodes. Terminating.....\n");
		exit(1);
	}
	if(kps_z != MAX_Z)		//If simulator is unable to handle more than MAX_Z key blocks per node
	{
		printf("No of keys in a block is not equal to MAX_Z. Check types.h file. Terminating.....\n");
		exit(1);
	}
	if(MAX_Z > 4)			//If simulator is unable to handle more than 4 key blocks per node
	{
		printf("assign_key_id function can't support MAX_Z more than 4. Check types.h file. Terminating.....\n");
		exit(1);
	}	
	
	kps_n = kps_b / kps_z;		//No of sensor nodes

	//allocate memeory for list fof key IDs. Size is list = no of sensor nodes
	key_list = (key_index *)calloc((kps_z * kps_n), sizeof(key_index));
	if(key_list == NULL)
	{
		perror("Error while allocating memory, assign_key_id(). Terminating.....\n");
		exit(1);	
	}
	for(i = 0; i < no_of_nodes ; i++)
	{
		fgets(msg, 100, key_fd);
		
		switch (kps_z)			//Extract up to 4 blocks
		{
			case 1:
				retvalue = sscanf(msg, "%d %d", tmp_key_id, (tmp_key_id + 1));
				key_list[i * kps_z].a = tmp_key_id[0];
				key_list[i * kps_z].b = tmp_key_id[1];
				key_list[i * kps_z].used = 0;
				break;
			case 2:
				retvalue = sscanf(msg, "%d %d %d %d", tmp_key_id, (tmp_key_id + 1), (tmp_key_id + 2), (tmp_key_id + 3));
				key_list[i * kps_z].a = tmp_key_id[0];
				key_list[i * kps_z].b = tmp_key_id[1];
				key_list[i * kps_z + 1].a = tmp_key_id[2];
				key_list[i * kps_z + 1].b = tmp_key_id[3];
				key_list[i * kps_z].used = 0;
				key_list[i * kps_z + 1].used = 0;
				break;
			case 3:
				retvalue = sscanf(msg, "%d %d %d %d %d %d", tmp_key_id, (tmp_key_id + 1), (tmp_key_id + 2), (tmp_key_id + 3), (tmp_key_id + 4), (tmp_key_id + 5));
				key_list[i * kps_z].a = tmp_key_id[0];
				key_list[i * kps_z].b = tmp_key_id[1];
				key_list[i * kps_z + 1].a = tmp_key_id[2];
				key_list[i * kps_z + 1].b = tmp_key_id[3];
				key_list[i * kps_z + 2].a = tmp_key_id[4];
				key_list[i * kps_z + 2].b = tmp_key_id[5];
				key_list[i * kps_z].used = 0;
				key_list[i * kps_z + 1].used = 0;
				key_list[i * kps_z + 2].used = 0;
				break;
			case 4:
				retvalue = sscanf(msg, "%d %d %d %d %d %d %d %d", tmp_key_id, (tmp_key_id + 1), (tmp_key_id + 2), (tmp_key_id + 3), (tmp_key_id + 4), (tmp_key_id + 5), (tmp_key_id + 6), (tmp_key_id + 7));
				key_list[i * kps_z].a = tmp_key_id[0];
				key_list[i * kps_z].b = tmp_key_id[1];
				key_list[i * kps_z + 1].a = tmp_key_id[2];
				key_list[i * kps_z + 1].b = tmp_key_id[3];
				key_list[i * kps_z + 2].a = tmp_key_id[4];
				key_list[i * kps_z + 2].b = tmp_key_id[5];
				key_list[i * kps_z + 3].a = tmp_key_id[6];
				key_list[i * kps_z + 3].b = tmp_key_id[7];
				key_list[i * kps_z].used = 0;
				key_list[i * kps_z + 1].used = 0;
				key_list[i * kps_z + 2].used = 0;
				key_list[i * kps_z + 3].used = 0;
				break;
			default:
				retvalue = sscanf(msg, "%d %d %d %d %d %d %d %d", tmp_key_id, (tmp_key_id + 1), (tmp_key_id + 2), (tmp_key_id + 3), (tmp_key_id + 4), (tmp_key_id + 5), (tmp_key_id + 6), (tmp_key_id + 7));
				key_list[i * kps_z].a = tmp_key_id[0];
				key_list[i * kps_z].b = tmp_key_id[1];
				key_list[i * kps_z + 1].a = tmp_key_id[2];
				key_list[i * kps_z + 1].b = tmp_key_id[3];
				key_list[i * kps_z + 2].a = tmp_key_id[4];
				key_list[i * kps_z + 2].b = tmp_key_id[5];
				key_list[i * kps_z + 3].a = tmp_key_id[6];
				key_list[i * kps_z + 3].b = tmp_key_id[7];
				key_list[i * kps_z].used = 0;
				key_list[i * kps_z + 1].used = 0;
				key_list[i * kps_z + 2].used = 0;
				key_list[i * kps_z + 3].used = 0;
				break;
		}		
	}

	//Assign key blocks randomly to nodes
	for (l = 0; l < NODESY ; l++)
	{
		for(m = 0 ; m < NODESX ; m++)
		{
			if(nodes[m][l].NID != 0)
			{
				while(1)
				{
					rnd_key = rand() % kps_n;			//Randomly pick a z blocks (node) & assign to a node
					if(key_list[rnd_key * kps_z].used == 0)	//If the first key block is not used
					{	
						switch (kps_z)			//Extract up to 4 blocks
						{
							case 1:
								key_list[rnd_key * kps_z].used = 1;		//Mark as used
								break;
							case 2:
								key_list[rnd_key * kps_z].used = 1;		//Mark as used
								key_list[rnd_key * kps_z + 1].used = 1;	//Mark as used
								break;
							case 3:
								key_list[rnd_key * kps_z].used = 1;		//Mark as used
								key_list[rnd_key * kps_z + 1].used = 1;	//Mark as used
								key_list[rnd_key * kps_z + 2].used = 1;	//Mark as used
								break;
							case 4:
								key_list[rnd_key * kps_z].used = 1;		//Mark as used
								key_list[rnd_key * kps_z + 1].used = 1;	//Mark as used
								key_list[rnd_key * kps_z + 2].used = 1;	//Mark as used
								key_list[rnd_key * kps_z + 3].used = 1;	//Mark as used
								break;
							default:
								key_list[rnd_key * kps_z].used = 1;		//Mark as used
								key_list[rnd_key * kps_z + 1].used = 1;	//Mark as used
								key_list[rnd_key * kps_z + 2].used = 1;	//Mark as used
								key_list[rnd_key * kps_z + 3].used = 1;	//Mark as used
								break;	
						}
						break;
					}
				}

				switch (kps_z)			//Extract up to 4 blocks
				{
					case 1:
						nodes[m][l].key_IDs[0] = key_list[rnd_key * kps_z];
						nodes[m][l].no_key_blocks = 1;
						break;
					case 2:
						nodes[m][l].key_IDs[0] = key_list[rnd_key * kps_z];
						nodes[m][l].key_IDs[1] = key_list[rnd_key * kps_z + 1];
						nodes[m][l].no_key_blocks = 2;
						break;
					case 3:
						nodes[m][l].key_IDs[0] = key_list[rnd_key * kps_z];
						nodes[m][l].key_IDs[1] = key_list[rnd_key * kps_z + 1];
						nodes[m][l].key_IDs[2] = key_list[rnd_key * kps_z + 2];
						nodes[m][l].no_key_blocks = 3;
						break;
					case 4:
						nodes[m][l].key_IDs[0] = key_list[rnd_key * kps_z];
						nodes[m][l].key_IDs[1] = key_list[rnd_key * kps_z + 1];
						nodes[m][l].key_IDs[2] = key_list[rnd_key * kps_z + 2];
						nodes[m][l].key_IDs[3] = key_list[rnd_key * kps_z + 3];
						nodes[m][l].no_key_blocks = 4;
						break;
					default:
						nodes[m][l].key_IDs[1] = key_list[rnd_key * kps_z + 1];
						nodes[m][l].key_IDs[0] = key_list[rnd_key * kps_z];
						nodes[m][l].key_IDs[2] = key_list[rnd_key * kps_z + 2];
						nodes[m][l].key_IDs[3] = key_list[rnd_key * kps_z + 3];
						nodes[m][l].no_key_blocks = 4;
						break;
				}
			}
		}
	}
}


/*------------------------------------------------------------------------------*
 * Following function count the no of nodes with a cluster head			*
 * use_kps 	- if 0 just count the no of nodes with a CH			*
 * 		- if 1 count the no of nodes with a CH due to unavailability	*
 * 		  of a common key with the CH					*
 * Retruns no of nodes with a CH						*
 *------------------------------------------------------------------------------*/
uint nodes_without_CH(uchar use_kps)
{
	uint x, y, no_ch, no_key;
	
	no_ch = 0;			//no of nodes with a CH
	no_key = 0;			//no of nodes with a CH due to lack of common key

	for (y = 0; y < NODESY ; y++)
	{
		for(x = 0 ; x < NODESX ; x++)
		{
			//node exist but not in a cluster
			if((nodes[x][y].NID != 0) && (nodes[x][y].CID == 0))
				no_ch++;
			
			//node exist but not in a cluster & due to lack of common key
			if((nodes[x][y].NID != 0) && (nodes[x][y].CID == 0) && (nodes[x][y].no_key_to_CH == 1))
				no_key++;
		}
	}

	if(use_kps == 1)		//no of nodes with a CH due to lack of common is key is needed
		return no_key;
	else				//else just return the no of nodes with a CH
		return no_ch;
}

/*------------------------------------------------------------------------------*
 * Following function print the no of common keys each node has with each & 	*
 * every node in its neighborhood to a file.					*
 *------------------------------------------------------------------------------*/
void print_no_common_keys()
{
	int minx, miny, maxx, maxy, common_keys, retvalue;
	uint x0, y0, x, y, nid, z, k, l;
	float distance;
	key_index key_parent[MAX_Z], key_child[MAX_Z]; 
	FILE *com_keyfd;
	
	com_keyfd = fopen(COM_KEYFILE, "w");
	if(com_keyfd == NULL)
	{
		perror("ERROR: ");
		exit(1);
	}

	for (y = 0; y < NODESY ; y++)
	{
		for(x = 0 ; x < NODESX ; x++)
		{		
			nid = nodes[x][y].NID;
			
			if(nid == 0)					//Skip if node doesn't exist
				continue;	
			x0 = (nid - 1) % NODESX;
			y0 = (nid - 1) / NODESX;
	
			minx = x0 - (R/GRIDX) - 1;                      // 1 hop neighbors are in the range of (X0 - R) to (X0 + R)
			if (minx < 0)					// Stay within the grid			
				minx = 0;
			maxx = x0 + (R/GRIDX) + 1;
			if(maxx >= NODESX)
				maxx = NODESX -1;

			miny = y0 - (R/GRIDY) - 1;                      // 1 hope neighbors are in the range of (Y0 - R) tp (Y0 + R)
			if (miny < 0)                                   // Stay within the grid
				miny = 0;		
			maxy = y0 + (R/GRIDY) + 1;
			if(maxy >= NODESY)
				maxy = NODESY - 1;

			for(z = 0; z < kps_z ; z++)
				key_parent[z] = nodes[x0][y0].key_IDs[z];

			for(l = miny; l <= maxy; l++)
			{
				for (k = minx; k <= maxx; k++)
				{
					distance = sqrt((k - x0)*(k - x0)*(GRIDX * GRIDX) + (l - y0)*(l - y0)*(GRIDY * GRIDY));
			
					// If within communication range & if the node exist & not me
					if ((distance <= R) && (nodes[k][l].NID != 0) && (distance != 0))     
					{
						for(z = 0; z < kps_z ; z++)
							key_child[z] = nodes[k][l].key_IDs[z];
						common_keys =  block_no_common_key(kps_z, key_parent, key_child, kps_k, kps_p);
						retvalue = sprintf(msg, "%d\n", common_keys);
						fputs(msg, com_keyfd);
					}
				}
			}
		}
	}
	fclose(com_keyfd);
}
