/***************************************************************************
 * File name : ccl.c 
 * Congestion Control Layer API 
 * Ahthor : Panho Lee 
 ***************************************************************************/
#include <math.h>
#include "ccl.h"
#include "onl.h"
#include "timer.h"

#define RA_STEP_SIZE 1
/**************************************************************
 * external functions
 **************************************************************/
extern int onl_send(int, char*, int, int);
extern unsigned int DEBUG;
extern int Random;
extern FILE *output;

/************************************************************/
static void  receive_conn_req(Session *, char *, int);
static void  receive_conn_ack(Session *, char *, int);
static void  receive_dat_pkt(Session*, Packet *, int);
static void  receive_dat_ack(Session*, char *, int);
static void  recv_data_packet_inode(Session * c, Packet *frame, int len);
static struct timespec * get_timeout(long, long);
static int read_packets(Session *, int *, int);
static void token_bucket_initialize(Session *c);
static void transmit_packet(Session *,  int );
static void insert_packet(Session *, Packet *, int );
static void latency(struct timeval *prev, Packet *last, int blkno);

/***************************************************************/
static Session *CurrentSession; /* current session */

static char  *temporaryBuffer[4][256]; /* 1MB temporary buffer */
static int  tempReceived[4];
static float total_packet_count = 0.0;
static int index = 0;
/****************************************************************/

timer_t mytimer;
static int timeOut;

void timer_intr(int sig, siginfo_t *extra, void *cruft);
void create_timer();

/**************************************************************************
 * create new session 
 * Input Parameters  myid - overlay node ID
 *                   myport - service UDP port
 *                   maxClient - maximum number of client 
 *                   maxBW - maximum bandwidth
 *                   mode - running mode (client, server, intermediate node)
 ***************************************************************************/
Session* ccl_session_create(int myid,  int myport, int maxClient, int maxBW, int mode)
{  
  CCLQueue *q;
  // allocate Session Data structure
  Session * session = (Session *) calloc(sizeof(Session), 1);

  printf("ccl_session_create(%d, %d, %d, %d, %d)\n", myid, myport, maxClient, maxBW, mode);
  if(session == NULL) {
    printf("Failed to allocate new session!\n");
    exit(0);
  }
  else {
    printf("New session is allocated [%x]! \n", (int ) session);
  }

  session->myID = myid;
  session->myport = myport;
  session->maxclient = maxClient;
  session->maxBW = maxBW;
  session->clientList = NULL;
  session->blk_time = 40000; /* 100 msec */
  session->mode = mode;
  session->clientList = NULL;

  sem_init(&session->clist_sem, 0, 1);

  /* create message queues */
  q = (CCLQueue *) malloc(sizeof(CCLQueue));
  session->Q_connect = ccl_init_queue(q, "connect");

  q = (CCLQueue *) malloc(sizeof(CCLQueue));
  session->Q_receive[0] = ccl_init_queue(q, "receive1");  
  q = (CCLQueue *) malloc(sizeof(CCLQueue));
  session->Q_receive[1] = ccl_init_queue(q, "receive2");  

  q = (CCLQueue *) malloc(sizeof(CCLQueue));
  session->Q_listen = ccl_init_queue(q, "listen"); 

  CurrentSession = session;
#if 1
  // launch intermediate node thread
  if(mode == I_NODE) {
    pthread_create(&session->PT_receive, NULL, run_inode, (void*)session);
  }
#endif

  return( session );
}

/************************************************************************
 * ccl_connect() : initiate connection
 * parameters : Session Id, destiantion IP address , Port number
 ************************************************************************/
int  ccl_connect
(
  Session * 		c,    /* Session ID   */
  int 			dst,  /* Destination Node ID */ 
  unsigned int		tr,   /* target rate  */
  unsigned int 		mr,   /* minimum rate */
  unsigned int 		ftype /* framing type */
)
{
  CCLQNode * init;
  CCLQNode * qnode;
  short    retry;
  
  CCLQueue* q = c->Q_connect;
  Packet *fr = (Packet *) malloc(128);

  // Set the server node ID. 
  c->dstID = dst;

  /* connection request */
  init = (CCLQNode *) calloc(sizeof(CCLQNode), 1);
  init->next = NULL;
  init->type = CONN_REQ;
  init->len = sizeof(COReq);

  fr->srcID = c->myID;
  fr->dstID = dst; 
  fr->opcode = CONN_REQ;
  fr->len = sizeof(COReq);
  fr->p.coreq.targetRate = tr;
  fr->p.coreq.minimumRate = mr;
  fr->p.coreq.framingType = ftype;
  init->message = (char *) fr;

  retry = 0;
  // send to overlay network layer
  onl_send(dst, init->message, fr->len+FRAME_HEADER_LEN, 0);
  // wait for acknowledgement 
  while (1) { 
    int rc = 0;
      /* wait for a message from socket receiver */
    pthread_mutex_lock(&q->mut);
    while(q->signal == 0) {  /* check for spurious wakeups */
      rc = pthread_cond_timedwait(&q->cond, &q->mut, get_timeout(1,0));         }
    q->signal = 0;

    if(rc != 0) {
      if(retry < 3) {
        onl_send(dst, init->message, fr->len+FRAME_HEADER_LEN, 0);
        retry++;
      }
      else {
        pthread_mutex_unlock(&q->mut); 
        if(DEBUG & 0x1) printf("Session request time-out!\n");
        ccl_destroy_node(init);
        return (-1);
      }
    }
      
    while (q->size > 0) {
      qnode = ccl_dequeue(q);
      pthread_mutex_unlock(&q->mut);  
      ccl_destroy_node(qnode);
      ccl_destroy_node(init);

      return(0);
    }
    pthread_mutex_unlock(&q->mut);  /* release start lock (it can go back to fgets) */
  }
  return -1; 
}

/************************************************************************
 * ccl_terminate() : terminate connection
 * parameters : Session Id, destiantion IP address , Port number
 ************************************************************************/
int  ccl_terminate
(
  Session * 		c,    /* Session ID   */
  int 			dst,  /* Destination Node ID */ 
  unsigned int		tr,   /* target rate  */
  unsigned int 		mr,   /* minimum rate */
  unsigned int 		ftype /* framing type */
)
{
  CCLQNode * init;
  CCLQNode * qnode;
  
  CCLQueue* q = c->Q_connect;
  Packet *fr = (Packet *) malloc(128);

  // Set the server node ID. 
  c->dstID = dst;

  /* connection request */
  init = (CCLQNode *) calloc(sizeof(CCLQNode), 1);
  init->next = NULL;
  init->type = CONN_REQ;
  init->len = sizeof(COReq);

  fr->srcID = c->myID;
  fr->dstID = dst; 
  fr->opcode = CONN_REQ;
  fr->len = sizeof(COReq);
  fr->p.coreq.targetRate = tr;
  fr->p.coreq.minimumRate = mr;
  fr->p.coreq.framingType = ftype;
  init->message = (char *) fr;

  // send to overlay network layer
  onl_send(dst, init->message, fr->len+FRAME_HEADER_LEN, 0);

  // wait for acknowledgement 
  while (1) { 
    int rc = 0;
      /* wait for a message from socket receiver */
    pthread_mutex_lock(&q->mut);
    while(q->signal == 0) {  /* check for spurious wakeups */
      rc = pthread_cond_timedwait(&q->cond, &q->mut, get_timeout(10,1000));         }
    q->signal = 0;

    if(rc != 0) {
      pthread_mutex_unlock(&q->mut); 
      if(DEBUG & 0x1) printf("Session request time-out!\n");
      ccl_destroy_node(init);
      return (-1);
    }
      
    while (q->size > 0) {
      qnode = ccl_dequeue(q);
      pthread_mutex_unlock(&q->mut);  
      ccl_destroy_node(qnode);
      ccl_destroy_node(init);

      return(0);
    }
    pthread_mutex_unlock(&q->mut);  /* release start lock (it can go back to fgets) */
  }
  return -1; 
}

int  ccl_recv(Session * c, char *buf, int bytes)
{
  CCLQNode* qnode;
  CCLQueue* q = c->Q_receive[0];
  Packet * dack;
  struct timeval now;
  struct timespec timeout;
  int    i, rc, done = 0;
  unsigned char tzdummy[64];

  /* to avoid memory copy .... */
  // c->buffer = buf;
  c->bytesReceived = 0;

  while (1) { 
      /* wait for a message from socket receiver */
    pthread_mutex_lock(&q->mut);
      /* get current time */
    gettimeofday(&now, (struct timezone *) &tzdummy);
    timeout.tv_sec = now.tv_sec + 5;
    timeout.tv_nsec = now.tv_usec * 1000; /* 1 micro-sec = 1000 nsec */
    rc = 1;
    while(q->signal == 0) {  /* check for spurious wakeups */
      rc = pthread_cond_timedwait(&q->cond, &q->mut, &timeout);    
      if(rc != 0) break;
    }

    switch(rc) {
      case 0:
        q->signal = 0;
        while (q->size > 0) {
         qnode = ccl_dequeue(q);
        }
        pthread_mutex_unlock(&q->mut);  /* release start lock (it can go back to fgets) */
        if(qnode->len < qnode->type) { /* packet loss */  
          // make ack message
          dack = (Packet *) malloc(sizeof(Packet));
          dack->opcode = DATA_ACK;
          dack->srcID = c->myID;
          dack->dstID = c->dstID;
          dack->len = sizeof(Packet);
          /* send data ACK */
          onl_send(dack->dstID, (char *) dack, dack->len, 0);
          free(dack);
        }
	// read data 
	// printf("Qnode Blkno %d received packet %d (%d) \n", qnode->blkno, tempReceived[qnode->blkno%4], qnode->blkno%4);
	c->bytesReceived = tempReceived[qnode->blkno%4];
	for(i = 0; i < tempReceived[qnode->blkno%4]; i++)
	  c->buffer[i] = temporaryBuffer[qnode->blkno%4][i];

	tempReceived[qnode->blkno%4] = 0;
	done = 1;
	break;

     default:
       if(rc == ETIMEDOUT) { /* time is up 5sec.*/
	 float elapsed;
	 printf("Done......!\n");
         elapsed = elapse_time(END, 0);
	 elapsed = elapsed - 5000.0;
         fprintf(output, "Total throughput %fMbps\n \n", (float ) (total_packet_count*1416.0)*8/(elapsed*1000.0)); 
	 return(0);
       }
       break;
    }

    if(done)
      return(qnode->blkno);
  }
  return -1; 
}

int ccl_listen_accept(Session *p)
{
  int  ret;
  ret = pthread_create(&p->PT_listen, NULL, run_listen, (void*)p);
  return ret; 
}

void* run_listen(void* P)
{
  CCLQNode* qnode;
  // int false_alarm = 1;
  static char foo[128];
  Session* p = (Session*)P;
  CCLQueue* q = p->Q_listen;  
  // CCLQueue* sq = Q_send;  

  while (1) {
      /* wait for a message from socket receiver */
    pthread_mutex_lock(&q->mut);
    while(q->signal == 0) {  /* check for spurious wakeups */
      pthread_cond_wait(&q->cond, &q->mut);    
    }

    q->signal = 0;

    while (q->size > 0) {
      Packet *fr;
      Packet *ack;

      qnode = ccl_dequeue(q);
      fr = (Packet *) qnode->message;

      // make ack message
      ack = (Packet *) foo;
      ack->srcID = p->myID;
      ack->dstID = fr->srcID;
      
      sem_wait(&p->clist_sem);
      ack->p.coack.result = addClient(p, (char *) fr);
      PrintList(p->clientList);
      sem_post(&p->clist_sem);
    
      ack->opcode = CONN_ACK;
      ack->len = 128; 
    
      /* send connection ACK */
      onl_send(ack->dstID, (char *) ack, 128, 0);
    }
    pthread_mutex_unlock(&q->mut);  /* release start lock (it can go back to fgets) */

    if(DEBUG & 0x1) printf("connection established....\n");
  }
  return (void*)NULL;
}



/*****************************************************************************
 * ccl_send() : transmit a block of data.
 * Input  : 
 * Output :
 *****************************************************************************/
int ccl_send
  (
  Session *	c, 	/* Session Id */
  int           blkno,
  char* 	blk,   /* block data */
  void(*selectFrame)(int, int, int, char *, PKTQueue*) /* frame selection callback */
  )
{
  ClientList * curr;
  int i, cnt = 0;
  int done = 0;
  Packet * frame;
  int nclient = 0;
  PKTNode *new_node;
  unsigned short bits;

  create_timer();
  /* application aware frame selection */
  sem_wait(&c->clist_sem);
  curr = c->clientList;
  while(curr != NULL) {
    bits = 0x1;
    curr->client.mask = 0;
    bits = bits << curr->client.currentRate;
    if(Random != 1) {
      for(i = 0; i < curr->client.targetRate - curr->client.currentRate;i++) {
        bits = bits << 1;
        curr->client.mask |= bits;
      }
    }
    else {  //Random
      bits = 0xffff;
      for(i = 0; i < curr->client.targetRate+1; i++) {
        bits = bits << 1;
        // curr->client.mask = curr->client.mask|bits;
      }
      curr->client.mask = (~bits); 
    }

  elapse_time(START, blkno);
    // printf("Target %d Current %d Mask %x \n",  curr->client.targetRate,  curr->client.currentRate, curr->client.mask);
    curr->client.packet_queue = (PKTQueue *) malloc(sizeof(PKTQueue));
    init_packet_queue(curr->client.packet_queue);
    selectFrame(curr->client.currentRate, curr->client.framingType, blkno, blk, curr->client.packet_queue);
    curr->client.pktcnt = curr->client.packet_queue->size;
    curr->client.sn = 0;
    curr = curr->Link;
    nclient++;
  }
  sem_post(&c->clist_sem);

  /* transmit packets to the clients */
  for(i = 0; i < TIME_SLOT_COUNT; i++) {
    int sent;

    if(i != 0) create_timer();

    curr = c->clientList;
    while(curr != NULL) {
       sent = 0;
       while((curr->client.pktcnt/TIME_SLOT_COUNT+1) > sent){

        new_node = dequeue_packet(curr->client.packet_queue);
        if(new_node != NULL) {
          frame = (Packet *) new_node->pkt; 
          if(Random)  {
            // curr->client.mask = ~curr->client.mask;
	    frame->pdp = new_node->bitpattern & curr->client.mask;
          }
          else {
	    frame->pdp = new_node->bitpattern | curr->client.mask;
          }

          frame->dstID = curr->client.nodeID.nodeId;
          frame->srcID = c->myID;
          frame->p.datpkt.sn = curr->client.sn++;
          frame->p.datpkt.pktcnt = curr->client.pktcnt;

          onl_send(frame->dstID, (char *) frame, new_node->len, curr->client.pathSW);
          curr->client.pathSW = 0; // clear path switch request 
          free(new_node);
	  cnt++;
	  sent++;
        }
        else {
          done++;
	  break;
        }
      }
      // printf("Time Slot %d Sent %d \n", i+1, sent);
      curr = curr->Link;
    }

    // spin timeout
    while(timeOut == 0) {
      // struct timespec nap = {1, 0};
      // nanosleep(&nap, NULL);
      pause(); // wait for timeout signal
    }
    // delete timer 
    timer_delete(mytimer);
  }

  if(done <  nclient )  {
    printf("Server slow-down\n");
  }

  // calculate new transmit rate (TRABOL )
  sem_wait(&c->clist_sem);
  curr = c->clientList;
  while(curr != NULL) {
    if(curr->client.acked == 0) { // no loss
      curr->client.currentRate = MIN((curr->client.currentRate + RA_STEP_SIZE),  (curr->client.targetRate) );   
    } 
    else {
      /* current path went bad */
      if(curr->client.currentRate == curr->client.minimumRate) {
        curr->client.pathSW = 1;
      }
      curr->client.currentRate = curr->client.minimumRate;
    }
    curr->client.acked = 0;
    curr = curr->Link;
  }
  sem_post(&c->clist_sem);

  // printf("Blk %d end of transmission %f \n", blkno, elapse_time(END, blkno));
  return(cnt);
}

void ccl_upcall_recv(unsigned char * msg, int len)
{
  Session *p = NULL;
  Packet *fr = (Packet *) msg;

  p = CurrentSession;
  if( CurrentSession == NULL) {
     printf("Session does not exist [Session %x]\n", (int) CurrentSession);
     exit(0);
  }
  switch(fr->opcode)
  {
    case CONN_REQ:
       receive_conn_req(p, msg, len);
       break;

    case CONN_ACK:
    case OVER_TRC_RSP:
printf("UPCALL %d \n", fr->opcode);
       receive_conn_ack(p, msg, len);
       break;

    case DATA_PKT:
      if(p->mode == C_NODE) {
        receive_dat_pkt(p, (Packet *)msg, len);
      } else if(p->mode == I_NODE) {
	recv_data_packet_inode(p, (Packet *) msg, len);
      }
      break;

    case DATA_ACK:
      receive_dat_ack(p, msg, len);
      break;
      /*
    case OVER_TRC_LAT:
      printf("Blk %d latency %f \n", fr->p.ovtrc.blkno, elapse_time(END, fr->p.ovtrc.blkno));
      free(msg);
      break;
      */
      

#if 0
     case CCL_TERMINATE_REQ:
       pthread_mutex_lock(&q->mut);   /* get the lock on the receive Q */
       // enqueue(q, qnode);             /* add packet from sock */
       q->signal = 1;                 /* set the signal flag to true */
       pthread_cond_signal(&q->cond); 
       pthread_mutex_unlock(&q->mut); /* unlock the receive Q*/
       break;

#endif
     default:
       break;
  }

  return;
}

static void receive_conn_ack(Session *p, char *msg, int len)
{
  CCLQNode * qnode;
  CCLQueue * q;

  qnode = (CCLQNode *) calloc(sizeof(CCLQNode), 1);

  qnode->message = malloc(len);
  memcpy(qnode->message, msg, len);
  free(msg);
  qnode->len = len;
  q = p->Q_connect; 
  pthread_mutex_lock(&q->mut);   /* get the lock on the receive Q */
  ccl_enqueue(q, qnode);         /* add packet from sock */
  q->signal = 1;                 /* set the signal flag to true */
  pthread_cond_signal(&q->cond); /* tell PT_listen that receive Q has data */
  pthread_mutex_unlock(&q->mut); /* unlock the receive Q*/

  return;
}

static void receive_conn_req(Session *p, char * msg, int len)
{
  CCLQNode * qnode;
  CCLQueue * q;

  qnode = (CCLQNode *) calloc(sizeof(CCLQNode), 1);

  qnode->message = malloc(len);
  memcpy(qnode->message, msg, len);
  free(msg);
  qnode->len = len;
  q = p->Q_listen; 
  pthread_mutex_lock(&q->mut);   /* get the lock on the receive Q */
  ccl_enqueue(q, qnode);         /* add packet from sock */
  q->signal = 1;                 /* set the signal flag to true */
  pthread_cond_signal(&q->cond); /* tell PT_listen that receive Q has data */
  pthread_mutex_unlock(&q->mut); /* unlock the receive Q*/

  return;
}

static void receive_dat_pkt(Session * c, Packet *frame, int len)
{
  CCLQNode * qnode;
  CCLQueue * q;
  static int blkno = -1;
  static unsigned short pattern;
  static int cnt = 0;
  static int pktno = 0; /* the number of packets for this block */
  static struct timeval start;
  static struct timeval prev;
  struct timeval end;
  struct timezone tz;
  float  elapsed = 1.0;

  total_packet_count = total_packet_count + 1.0;
  if(blkno == -1) { /* is the first block */
    blkno = frame->p.datpkt.blkno;
    // pktno = frame->p.datpkt.pktcnt;
    gettimeofday(&start, &tz);
    elapse_time(START, 0);
  }

  // simply check block number ... 
  // if( (frame->p.datpkt.blkno != blkno) || (cnt == pktno)) { /* is new block arrived ? */
  if( frame->p.datpkt.blkno != blkno) { /* is new block arrived ? */
    // latency(&prev, frame, blkno);
    gettimeofday(&end, &tz);
    qnode = (CCLQNode *) calloc(sizeof(CCLQNode), 1);
    qnode->len = cnt;    /* number of packets received */
    if(pattern && 0xf000) 
      qnode->type = pktno+1; /* total number of packet transmitted */
    else 
      qnode->type = 10000; // a big number 

    qnode->blkno = blkno;
    // tempReceived[blkno%4] = 0;
    q = c->Q_receive[0]; 
    pthread_mutex_lock(&q->mut);   /* get the lock on the receive Q */
    ccl_enqueue(q, qnode);         /* add packet from sock */
    q->signal = 1;                 /* set the signal flag to true */
    pthread_cond_signal(&q->cond); /* tell PT_listen that receive Q has data */
    pthread_mutex_unlock(&q->mut); /* unlock the receive Q*/
    elapsed = (end.tv_sec - start.tv_sec) * 1000000.0 + (end.tv_usec - start.tv_usec)*1.0;
    // printf("Block No. %d [%d/%d] Elapsed Time [%fmsec] Throughput %fMbps\n", 
    //	    blkno, cnt, pktno+1, elapsed/1000.0, (float ) (cnt*1460.0)*8/(elapsed)); 

    // printf("Block No. %d [%d/%d] Elapsed Time [%fmsec] \n", blkno, cnt, pktno+1, elapsed/1000.0); 
    // new block number
    blkno = frame->p.datpkt.blkno;
    pktno = frame->p.datpkt.pktcnt;
    cnt = 0;
    gettimeofday(&start, &tz); /* get new start time */
    cnt++;
    temporaryBuffer[blkno%4][tempReceived[blkno%4]] = (char *) frame;
    tempReceived[blkno%4]++;
  }
  else {
    // copy user data packet to the application buffer
    cnt++;
    pktno = frame->p.datpkt.sn;
    pattern = frame->pdp;

    temporaryBuffer[blkno%4][tempReceived[blkno%4]] = (char *) frame;
    tempReceived[blkno%4]++;
  }

  gettimeofday(&prev, &tz);
  // free(frame);
  return;    
}

static void receive_dat_ack(Session * c, char *msg, int len)
{
  Packet * frame = (Packet *) msg;
  ClientList   * curr;

  printf("Data ACK received \n");
  sem_wait(&c->clist_sem);
  curr = c->clientList;
  curr = SearchList(frame->srcID, curr);
  if(curr != NULL)
    curr->client.acked = 1;
  else
    printf("Unknown client %d \n", frame->srcID);
  sem_post(&c->clist_sem);

  free(msg);

  return;
}

static struct timespec * get_timeout(long sec, long usec)
{
  static struct timespec timeout;
  struct timeval current;
  unsigned char tzdummy[64];

  gettimeofday(&current, (void *) tzdummy);
  timeout.tv_sec = current.tv_sec + sec;
  timeout.tv_nsec = current.tv_usec * usec;

  return(&timeout);
}

void create_timer()
{
  struct itimerspec i;
  struct sigaction sa;
  struct sigevent timer_event;

  timeOut = 0;

  sigemptyset(&sa.sa_mask);
  sa.sa_flags=SA_SIGINFO;         /*real-time signal*/
  sa.sa_sigaction=timer_intr;     /*pointer to action*/

  if(sigaction(SIGRTMIN, &sa, NULL) < 0){
    perror("sigaction error");
    exit(1);
  }
        
  /*
   * create a timer based upon the CLOCK_REALTIME clock 
   */
  i.it_interval.tv_sec=0;
  i.it_interval.tv_nsec=0;
  i.it_value.tv_sec = 0;
  // i.it_value.tv_nsec = 20000000; //20 msec
  i.it_value.tv_nsec = 18500000; //20 msec

  timer_event.sigev_notify=SIGEV_SIGNAL;
  timer_event.sigev_signo= SIGRTMIN;
  timer_event.sigev_value.sival_ptr = (void *)&mytimer;

  if (timer_create(CLOCK_REALTIME, &timer_event, &mytimer) < 0)
      perror("timer create error");

  /* relative timer, go off at the end of the interval*/
  if(timer_settime(mytimer, 0 , &i, NULL) < 0){
     perror("settimer");
  }
    
}

void timer_intr(int sig, siginfo_t *extra, void *cruft)
{
  timeOut = 1;
  // printf("TIME OUT \n");
  return;
}

/*******************************************************************************
 *
 * Intermediate Node Implementation (Token Bucket).
 *
 ******************************************************************************/
void *run_inode(void *P)
{
  Session *p = (Session *) P;
  int bktcnt = 0;
  int eob = 0;
  static int blkno = -1;
  static  struct timespec nap = {0, 20000000};

  while(1) {
    eob = 0;
    switch(bktcnt)
    { 
      case 0: // 1st. time slot 
        // create_timer();
//    elapse_time(START);
        index++;
        if(index >= p->nclient) index = 0;
        token_bucket_initialize(p);
	// printf("Packet inserted %d \n", pktcount);
	eob = read_packets(p, &blkno, 1);
	transmit_packet(p, bktcnt);
        // while(timeOut == 0) { pause(); } /* spin */
        // timer_delete(mytimer);           /* delete timer */
	if(eob == EOB) {
	  bktcnt =0;
	}
	else bktcnt++;
	break;

      case TIME_SLOT_COUNT-2:
	// create_timer();
    nanosleep(&nap, NULL);
    nanosleep(&nap, NULL);
	eob = read_packets(p, &blkno, 1);
	transmit_packet(p, bktcnt);
	if(eob == EOB)   {
	  bktcnt = 0;
          // timer_delete(mytimer);           /* delete timer */
	}
	else {
          // while(timeOut == 0) { pause(); } /* spin */
          //  timer_delete(mytimer);           /* delete timer */
	  bktcnt++;
	}
	break;

      case TIME_SLOT_COUNT-1: // last time slot
	// create_timer();
    nanosleep(&nap, NULL);
	eob = read_packets(p, &blkno, 1);
	transmit_packet(p, bktcnt);
 //   elapse_time(END);
        // while(timeOut == 0) { pause(); } /* spin */
        // timer_delete(mytimer);           /* delete timer */

	while(eob != EOB) 
	  eob = read_packets(p, &blkno, 0);
	
	// printf("force to recieve next block %d\n", blkno);

	bktcnt = 0;
	break;

      default:
	// create_timer();
    nanosleep(&nap, NULL);
	eob = read_packets(p, &blkno, 1);
	transmit_packet(p, bktcnt);
	if(eob == EOB)   {
// printf("new block start %d (count %d)\n", blkno, bktcnt);
	  bktcnt = 0;
          // timer_delete(mytimer);           /* delete timer */
	}
	else {
          // while(timeOut == 0) { pause(); } /* spin */
          //  timer_delete(mytimer);           /* delete timer */
	  bktcnt++;
	}
	break;
    }
  }
}

static int read_packets(Session *p, int *blkno, int s) 
{
  CCLQueue* q; 
  Packet *frame;
  CCLQNode * qnode;

  q  = p->Q_receive[0]; 
  pthread_mutex_lock(&q->mut);
  while(q->signal == 0) { /* check for spurious wakeups */
    pthread_cond_wait(&q->cond, &q->mut);
  }
  q->signal = 0;
  /* copy received packet to the client packet queue */
  while(q-> size > 0) {
    qnode = ccl_dequeue(q);
    if(qnode->type == EOB) {
      free(qnode);

      *blkno = qnode->len;
      pthread_mutex_unlock(&q->mut);
      return (EOB);
    }

    frame = (Packet *) qnode->message;
    if(*blkno == -1)  {
      *blkno = frame->p.datpkt.blkno;
    }


    if(*blkno == frame->p.datpkt.blkno && s != 0) {
  //  printf("RECV %d / %d \n", frame->p.datpkt.blkno, frame->p.datpkt.sn);
      // insert the received packets. 
      insert_packet(p, (Packet *) qnode->message, qnode->len);
    }

    free(qnode->message);
    free(qnode);
  }
  pthread_mutex_unlock(&q->mut);

  return 0;
}

static void insert_packet(Session *p, Packet *frame, int len)
{
  ClientList * curr;
  PKTNode *newnode;
  char  *foo;

  curr = p->clientList;
  while(curr != NULL) {
    /* check bitmask*/
    if(Random == 1) {
      if(curr->client.cnt <= curr->client.pktcnt)
      {
        foo = malloc(len);
        memcpy(foo, (char *) frame, len);
        newnode = init_packet_node(foo, len);
        enqueue_packet(curr->client.packet_queue, newnode);
        curr->client.cnt++;
      }
    }
    else {
      if((curr->client.mask & frame->pdp) && (curr->client.cnt <= curr->client.pktcnt))
      {
        foo = malloc(len);
        memcpy(foo, (char *) frame, len);
        newnode = init_packet_node(foo, len);
        enqueue_packet(curr->client.packet_queue, newnode);
        curr->client.cnt++;
      }
    }
    curr = curr->Link;
  }
}


static void transmit_packet(Session *p, int bktcnt)
{
  ClientList * curr;
  Packet * frame;
  register unsigned short mask;
  int cnt = 0,i;

  /* transmit packets to the end clients */
  curr = p->clientList;
  while(curr != NULL) {
	  mask = curr->client.token_bucket[bktcnt];
    while( curr->client.token_bucket[bktcnt]  > 0) {
// printf("Client ID %d bucket %d size %d\n", curr->client.nodeID.nodeId,  bktcnt, curr->client.token_bucket[bktcnt]  );
      PKTNode * new_node = dequeue_packet(curr->client.packet_queue);
      if(new_node != NULL) {
        frame = (Packet *) new_node->pkt; 
        frame->dstID = curr->client.nodeID.nodeId;
        frame->srcID = p->myID;
        frame->p.datpkt.sn = curr->client.sn++;
        frame->p.datpkt.pktcnt = curr->client.pktcnt;
	if(bktcnt==TIME_SLOT_COUNT-1 && curr->client.token_bucket[bktcnt]==1)
	  frame->pdp |= 0xf000; // last packet for this block

        onl_send(frame->dstID, (char *) frame, new_node->len, curr->client.pathSW);
	i = frame->p.datpkt.blkno;
        curr->client.token_bucket[bktcnt]--;
        curr->client.pathSW = 0; // clear path switch request 
        free(new_node->pkt);
        free(new_node);
	cnt++;
      }
      else break; 
    }

    // bucket size update
    mask = (unsigned short) 1;
    if(bktcnt != TIME_SLOT_COUNT-1) {
      curr->client.token_bucket[bktcnt+1] += curr->client.token_bucket[bktcnt];
      mask = mask << curr->client.currentRate;
      curr->client.mask = mask;

      // steal heuristic 
      if(curr->client.token_bucket[bktcnt] > 0) {
        curr->client.mask = curr->client.mask | (mask*2);
        curr->client.mask = curr->client.mask | (mask*4);
      }

      
#if 0 
      cnt = (int) (ceilf( ((float)curr->client.token_bucket[bktcnt+1] - 
                          curr->client.pktcnt/(float)TIME_SLOT_COUNT ) /
                         (float) (TIME_SLOT_COUNT - bktcnt) ) );
      for(i = 0; i < MIN(cnt, 15); i++) 
      {
        mask = mask << 1;
        curr->client.mask |= mask;
      }
#endif
      // printf("steal count %d bucket no. %d mask %x\n", cnt, bktcnt+1, curr->client.mask);
    }
    curr = curr->Link;
  }
}

/*
 * calculate new transmit rate for each client.
 * Initialize token buckets
 */
static void token_bucket_initialize(Session *c)
{
  int  bkt;
  float total_packet;
  ClientList * curr;
  PKTNode * new_node;

  // calculate new transmit rate (TRABOL )
  sem_wait(&c->clist_sem);
  curr = c->clientList;

  while(curr != NULL) {
    /* clear client's packet queue */
    bkt = 0;
    while(1) {
      new_node = dequeue_packet(curr->client.packet_queue);
      if(new_node != NULL) {
#if 0
        Packet *frame;

        frame = (Packet *) new_node->pkt; 
        frame->dstID = curr->client.nodeID.nodeId;
        frame->srcID = c->myID;
        frame->p.datpkt.sn = curr->client.sn++;
        frame->p.datpkt.pktcnt = curr->client.pktcnt;
        onl_send(frame->dstID, (char *) frame, new_node->len, curr->client.pathSW);
#endif
	free(new_node->pkt);
	free(new_node);
      }
      else break;
    } 
    if(curr->client.acked == 0) { // no loss
      curr->client.currentRate = MIN((curr->client.currentRate + RA_STEP_SIZE),  (curr->client.targetRate) );   
    } 
    else {
    /* current path went bad */
      if(curr->client.currentRate == curr->client.minimumRate) {
        curr->client.pathSW = 1;
      }
      curr->client.currentRate = curr->client.minimumRate;
    }

    // 1433 = (2*1460 + 1380)/3
    // total_packet = floorf((((float) curr->client.currentRate/10.0) * 2097152.0)/(1433.0 * 8.0));
    // total_packet = ceilf((((float) curr->client.currentRate/MAX_RATE) * 192.0));
    total_packet = floor((((float) curr->client.currentRate/MAX_RATE) * 64.0)-1);
    total_packet = total_packet * 3;
    // printf("total packet %f client ID %d current Rate %d\n", total_packet, curr->client.nodeID.nodeId, curr->client.currentRate);

    //printf("total packet %f \n", total_packet);
    // token bucket initialize based on current sending rate.
    curr->client.pktcnt = (int ) total_packet;
    curr->client.cnt = (int ) 0;
    for(bkt = 0; bkt < TIME_SLOT_COUNT-1; bkt++) {
      curr->client.token_bucket[bkt] = (int ) floorf(total_packet/(float) (TIME_SLOT_COUNT-2)+0.5);
      /*
      curr->client.token_bucket[bkt] = (int ) floorf(total_packet/(float) TIME_SLOT_COUNT);
      */
	// curr->client.pktcnt += curr->client.token_bucket[bkt];
    }
    curr->client.token_bucket[TIME_SLOT_COUNT-2] = (int ) 0;
    curr->client.token_bucket[TIME_SLOT_COUNT-1] = (int ) 0;
    curr->client.acked = 0;
    curr->client.sn = 0;

    curr = curr->Link;
  }
  sem_post(&c->clist_sem);
  return;
}

static void recv_data_packet_inode(Session * c, Packet *frame, int len)
{
  CCLQNode * qnode;
  CCLQueue * q;
  static int blkno = -1;
  static int pktno = 0; /* the number of packets for this block */
  int    eob = 0;  // end of block flag
  static int cnt = 0;  /* packet count */
  static char foo[128];

  q = c->Q_receive[0]; 
  if(blkno == -1) { /* is the first block */
    blkno = frame->p.datpkt.blkno;
    pktno = frame->p.datpkt.pktcnt;
    elapse_time(START, 0);
  }
  total_packet_count++;

  if(frame->p.datpkt.blkno == 1005) {
    float elapsed;

    printf("Done......!\n");
    elapsed = elapse_time(END, 0);
    elapsed = elapsed - 5000.0;
    // printf("Total throughput %fMbps\n", (float ) (total_packet_count*1420.0)*8/(elapsed*1000.0));
    exit(0);
  }

  qnode = (CCLQNode *) calloc(sizeof(CCLQNode), 1);
  qnode->len = len;    /* number of packets received */
  qnode->message = (char *) frame;

// printf("RECV : blk %d sn %d pdp %x\n", frame->p.datpkt.blkno, frame->p.datpkt.sn, frame->pdp); 
  // simply check block number ... 
  if(frame->p.datpkt.blkno != blkno) { /* is new block arrived ? */
    // insert end-of-block 
    eob = EOB; 

    if(cnt != pktno) { // packet loss 
       Packet *dack;
      // send feedback
       dack = (Packet *) foo;
       dack->opcode = DATA_ACK;
       dack->srcID = c->myID;
       dack->dstID = c->dstID;
       printf("ACK destination %d \n", dack->dstID);
       dack->len = 128;
       /* send data ACK */
       onl_send(dack->dstID, (char *) dack, dack->len, 0);
    }
    blkno = frame->p.datpkt.blkno;
    pktno = frame->p.datpkt.pktcnt;
    cnt = 0;
  }

  pthread_mutex_lock(&q->mut);   /* get the lock on the receive Q */
  if(eob == EOB) {
    CCLQNode *eob_node = (CCLQNode *) calloc(sizeof(CCLQNode), 1);
    eob_node->type = EOB;
    eob_node->len = blkno; // new block number
    ccl_enqueue(q, eob_node);
  }
  ccl_enqueue(q, qnode);         /* add packet from sock */
  q->signal = 1;                 /* set the signal flag to true */
  pthread_cond_signal(&q->cond); /* tell PT_listen that receive Q has data */
  pthread_mutex_unlock(&q->mut); /* unlock the receive Q */
  cnt++;

  return;    
}

Packet *overlay_trace(int myID, int myport, Packet *pkt)
{
  Packet * reply;
  static char foo[128];

  reply = (Packet *) foo;

  reply->srcID = myID;
  reply->dstID = pkt->srcID;
  reply->opcode = OVER_TRC_RSP;
  if(CurrentSession != NULL)
    reply->p.ovtrc.nodeType = CurrentSession->mode;
  else
    reply->p.ovtrc.nodeType = R_NODE;
	
  reply->p.ovtrc.port = myport;
  reply->p.ovtrc.dstID = pkt->p.ovtrc.dstID;

  return (reply);
}

static void latency(struct timeval *prev, Packet *last, int blkno)
{
  struct timeval now;
  static char foo[128];
  struct timezone tz;
  Packet *fr;

  gettimeofday(&now, &tz);

  fr = (Packet *) foo;
  fr->srcID = last->dstID;
  fr->dstID = last->srcID;
  fr->opcode = OVER_TRC_LAT;
  fr->len = 128;

  fr->p.ovtrc.t_sec = now.tv_sec - prev->tv_sec;
  fr->p.ovtrc.t_usec = now.tv_usec - prev->tv_usec;
  fr->p.ovtrc.blkno = blkno;

  onl_send(fr->dstID, (char *) fr, fr->len, 0);
}

#if 0
static unsigned long freq[10];

static void packet_count(Packet *pkt)
{
   if(pkt->pdp & 0x400) freq[9]++;
   if(pkt->pdp & 0x200) freq[8]++;
   if(pkt->pdp & 0x100) freq[7]++;
   if(pkt->pdp & 0x80) freq[6]++;
   if(pkt->pdp & 0x40) freq[5]++;
   if(pkt->pdp & 0x20) freq[4]++;
   if(pkt->pdp & 0x10) freq[3]++;
   if(pkt->pdp & 0x8) freq[2]++;
   if(pkt->pdp & 0x4) freq[1]++;
   if(pkt->pdp & 0x2) freq[0]++;
}

void distrib()
{
  printf("Total number of packets %d \n", pkt_cnt);
  printf("Rate 1Mbps : %d \n", freq[0]);
  printf("Rate 2Mbps : %d \n", freq[1]);
  printf("Rate 3Mbps : %d \n", freq[2]);
  printf("Rate 4Mbps : %d \n", freq[3]);
  printf("Rate 5Mbps : %d \n", freq[4]);
  printf("Rate 6Mbps : %d \n", freq[5]);
  printf("Rate 7Mbps : %d \n", freq[6]);
  printf("Rate 8Mbps : %d \n", freq[7]);
  printf("Rate 9Mbps : %d \n", freq[8]);
  printf("Rate 10Mbps : %d \n", freq[9]);
}
#endif
